/**
 * 
 *
 * Software: LGExtract
 * Authors: Matthieu Constant and Elsa Tolone
 *
 * Copyright (C) 2010-2011 Université Paris-Est Marne-la-Vallée
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 *
 */


package fr.umlv.lgextract;

import fr.umlv.lgextract.actions.AddAction;
import fr.umlv.lgextract.actions.CreateAction;
import fr.umlv.lgextract.actions.LGExtractActions;
import fr.umlv.lgextract.actions.LGExtractPath;
import fr.umlv.lgextract.actions.NamedActions;
import fr.umlv.lgextract.boolexp.BooleanAndNode;
import fr.umlv.lgextract.boolexp.BooleanLeaf;
import fr.umlv.lgextract.boolexp.BooleanNode;
import fr.umlv.lgextract.boolexp.BooleanNotNode;
import fr.umlv.lgextract.boolexp.BooleanOrNode;
import fr.umlv.lgextract.boolexp.BooleanParNode;
import fr.umlv.lgextract.exceptions.LGExtractException;
import fr.umlv.lgextract.exceptions.NotFoundLingObjException;
import fr.umlv.lgextract.lobjects.FeatureStructure;
import fr.umlv.lgextract.lobjects.LGExtractLingObjects;
import fr.umlv.lgextract.lobjects.LingList;
import fr.umlv.lgextract.lobjects.LingObj;
import fr.umlv.lgextract.lobjects.StringFeature;
import fr.umlv.lgextract.tools.ActionInstruction;
import fr.umlv.lgextract.tools.Add;
import fr.umlv.lgextract.tools.Add_Conditional;
import fr.umlv.lgextract.tools.And;
import fr.umlv.lgextract.tools.Att_Star;
import fr.umlv.lgextract.tools.Attrib;
import fr.umlv.lgextract.tools.Create;
import fr.umlv.lgextract.tools.Create_Conditional;
import fr.umlv.lgextract.tools.Define;
import fr.umlv.lgextract.tools.Define_Star;
import fr.umlv.lgextract.tools.Elem;
import fr.umlv.lgextract.tools.Element_Star;
import fr.umlv.lgextract.tools.Empty_List;
import fr.umlv.lgextract.tools.Empty_Simplefs;
import fr.umlv.lgextract.tools.Feature_Fs;
import fr.umlv.lgextract.tools.Feature_Fs_Inherits;
import fr.umlv.lgextract.tools.Feature_List;
import fr.umlv.lgextract.tools.Feature_Name;
import fr.umlv.lgextract.tools.Feature_Value;
import fr.umlv.lgextract.tools.Features_Feature;
import fr.umlv.lgextract.tools.Fs_Inherits;
import fr.umlv.lgextract.tools.Fs_Simple;
import fr.umlv.lgextract.tools.Header_Module;
import fr.umlv.lgextract.tools.IInstruction;
import fr.umlv.lgextract.tools.Info;
import fr.umlv.lgextract.tools.Information_Star;
import fr.umlv.lgextract.tools.Infos;
import fr.umlv.lgextract.tools.Instruction_Star;
import fr.umlv.lgextract.tools.List;
import fr.umlv.lgextract.tools.Not;
import fr.umlv.lgextract.tools.Not_Propactionname;
import fr.umlv.lgextract.tools.Obj_Feature;
import fr.umlv.lgextract.tools.Obj_Fs;
import fr.umlv.lgextract.tools.Obj_List;
import fr.umlv.lgextract.tools.Or;
import fr.umlv.lgextract.tools.OtherActionName;
import fr.umlv.lgextract.tools.OtherPropCall;
import fr.umlv.lgextract.tools.Parenthesis;
import fr.umlv.lgextract.tools.Prop;
import fr.umlv.lgextract.tools.Prop_Value;
import fr.umlv.lgextract.tools.Propactionname;
import fr.umlv.lgextract.tools.Propactionname_Always;
import fr.umlv.lgextract.tools.Property_Star;
import fr.umlv.lgextract.tools.Several_Features;
import fr.umlv.lgextract.tools.Several_Path;
import fr.umlv.lgextract.tools.Simple_Path;
import fr.umlv.lgextract.tools.Simplefs;
import fr.umlv.lgextract.tools.Start;
import fr.umlv.lgextract.tools.Value;
import fr.umlv.lgextract.tools.Value_Bool;
import fr.umlv.lgextract.tools.Value_Id;
import fr.umlv.lgextract.tools.Visitor;

/**
 * Represents the visitor traversing the AST corresponding to the script 
 * 
 * @author Matthieu Constant
 *
 */


public class LGExtractVisitor extends Visitor<Void,LGExtractVisitorMemory,Void,Throwable> {
   	private final LGExtractActions actions;
    private final LGExtractLingObjects objects = new LGExtractLingObjects();;
    private final LGExtractFormatting formats = new LGExtractFormatting();
    
    public LGExtractVisitor(){
    	actions = new LGExtractActions(objects);
    }
        

    
    /*FORMATTING*/
	

	@Override
	public Void visit(Att_Star att_star, LGExtractVisitorMemory param) throws Throwable {
		for(Attrib a:att_star.nodeList()){
			a.accept(this, param);
		}
		return null;
	}


	@Override
	public Void visit(Attrib attrib, LGExtractVisitorMemory param) throws Throwable {		
		formats.addAttribute(param.getCurrentElement(), attrib.getId());
		return null;
	}


	@Override
	public Void visit(Elem elem, LGExtractVisitorMemory param) throws Throwable {
		String elt = elem.getId2();
		param.setCurrentElement(elt);
		formats.addElement(elem.getId(),elt);
		elem.getAtt_star().accept(this, param);
		return null;
	}

	
	@Override
	public Void visit(Element_Star element_star, LGExtractVisitorMemory param) throws Throwable {
		for(Elem e:element_star.nodeList()){
			e.accept(this, param);
		}
		return null;
	}
	
	/* HEADER */
	
	@Override
	public Void visit(Header_Module header_module, LGExtractVisitorMemory param) throws Throwable {
		return null;
	}
	
	/* INFO */
	
	@Override
	public Void visit(Info info, LGExtractVisitorMemory param) throws Throwable {
		return null;
	}

	
	@Override
	public Void visit(Information_Star information_star, LGExtractVisitorMemory param) throws Throwable {
		return null;
	}

	@Override
	public Void visit(Infos infos, LGExtractVisitorMemory param) throws Throwable {
		return null;
	}

	/* BOOLEXP */
		
	@Override
	public Void visit(And and, LGExtractVisitorMemory param) throws Throwable {
		and.getBoolexp().accept(this, param);
		and.getBoolexp2().accept(this, param);
		BooleanNode n2 = param.popBoolExp();
		BooleanNode n1 = param.popBoolExp();
		param.pushBoolExp(new BooleanAndNode(n1,n2));
		return null;
	}
	
	@Override
	public Void visit(Not not, LGExtractVisitorMemory param) throws Throwable {
		not.getBoolexp().accept(this, param);
		BooleanNode n = param.popBoolExp();
		param.pushBoolExp(new BooleanNotNode(n));
		return null;
	}

	@Override
	public Void visit(Or or, LGExtractVisitorMemory param) throws Throwable {
		or.getBoolexp().accept(this, param);
		or.getBoolexp2().accept(this, param);
		BooleanNode n2 = param.popBoolExp();
		BooleanNode n1 = param.popBoolExp();
		param.pushBoolExp(new BooleanOrNode(n1,n2));
		return null;
	}

	@Override
	public Void visit(Parenthesis parenthesis, LGExtractVisitorMemory param) throws Throwable {
		parenthesis.getBoolexp().accept(this, param);
		BooleanNode n = param.popBoolExp();
		param.pushBoolExp(new BooleanParNode(n));
		return null;
	}	

	@Override
	public Void visit(Value value, LGExtractVisitorMemory param) throws Throwable {
	    value.getBoolvalue().accept(this, param);
		return null;
	}
	
	
	@Override
	public Void visit(Value_Bool value_bool, LGExtractVisitorMemory param) throws Throwable {
		param.pushBoolExp(new BooleanLeaf(value_bool.getVal()));
		return null;
	}
	
	@Override
	public Void visit(Prop_Value prop_value, LGExtractVisitorMemory param) throws Throwable {
		param.pushBoolExp(new BooleanLeaf(prop_value.getPropname()));
		return null;
	}
	
	@Override
	public Void visit(Value_Id value_id, LGExtractVisitorMemory param) throws Throwable {		
		param.pushBoolExp(new BooleanLeaf(value_id.getId()));
		return null;
	}
	
	
	/*PROPERTIES*/
	

	@Override
	public Void visit(Property_Star property_star, LGExtractVisitorMemory param) throws Throwable {
		for(Prop p : property_star.nodeList()){
			p.accept(this, param);
		}	
		return null;
	}


	
	@Override
	public Void visit(Prop prop, LGExtractVisitorMemory param) throws Throwable {
		NamedActions na = new NamedActions(param.getActionName());		
		param.setNamedActions(na);
		prop.getActionname().accept(this, param);
		
		prop.getInstruction_star().accept(this,param);
		actions.addAction(na);
		if(na.getName() == null){
			System.err.println(na);
		}
		return null;
	}

	
	@Override
	public Void visit(Instruction_Star instruction_star, LGExtractVisitorMemory param) throws Throwable {
		
		
		for(IInstruction inst:instruction_star.nodeList()){
			inst.accept(this, param);
		}		
		return null;
	}


	@Override
	public Void visit(OtherActionName otherActionName, LGExtractVisitorMemory param) throws Throwable {
		//param.setActionName(otherActionName.getId());
		param.getNamedActions().setActionName(otherActionName.getId());
		return null;
	}

	@Override
	public Void visit(Propactionname propactionname, LGExtractVisitorMemory param) throws Throwable {
		param.getNamedActions().setActionName(propactionname.getPropname());
		//param.setActionName(propactionname.getPropname());
		return null;
	}

	@Override
	public Void visit(Not_Propactionname not_propactionname, LGExtractVisitorMemory param) throws Throwable {
		//param.setActionName(not_propactionname.getPropname());	
		param.getNamedActions().setActionName(not_propactionname.getPropname());
		param.getNamedActions().setNot(true);
		return null;
	}


	@Override
	public Void visit(ActionInstruction actionInstruction, LGExtractVisitorMemory param) throws Throwable {
		actionInstruction.getAction().accept(this, param);
		return null;
	}



	@Override
	public Void visit(OtherPropCall otherPropCall, LGExtractVisitorMemory param) throws Throwable {
		String name = otherPropCall.getId();
		NamedActions a = actions.getNamedActions(name);
		if(a == null){
			throw new NotFoundLingObjException("Property "+name);
		}
		 param.getNamedActions().addAction(a);
		return null;
	}

	
	
	@Override
	public Void visit(Several_Path several_path, LGExtractVisitorMemory param) throws Throwable {
		param.getCurrentPath().add(several_path.getId());
		several_path.getPath().accept(this, param);
		return null;
	}


	@Override
	public Void visit(Simple_Path simple_path, LGExtractVisitorMemory param) throws Throwable {
		param.getCurrentPath().add(simple_path.getId());
		return null;
	}

	
	@Override
	public Void visit(Add add, LGExtractVisitorMemory param) throws Throwable {
		LGExtractPath path = new LGExtractPath(objects);
		param.setCurrentPath(path);
		String id = add.getId();
		LingObj o = objects.getByName(id);
		if(o == null){
			throw new NotFoundLingObjException(id);
		}
		add.getPath().accept(this, param);
		param.getNamedActions().addAction(new AddAction(path,o, new BooleanLeaf("true")));
		return null;
	}
	
	@Override
	public Void visit(Add_Conditional add_conditional, LGExtractVisitorMemory param) throws Throwable {
		LGExtractPath path = new LGExtractPath(objects);
		param.setCurrentPath(path);
		String id = add_conditional.getId();
		LingObj o = objects.getByName(id);
		if(o == null){
			throw new NotFoundLingObjException(id);
		}
		add_conditional.getPath().accept(this, param);
		add_conditional.getBoolexp().accept(this, param);
		param.getNamedActions().addAction(new AddAction(path,o, param.peekBoolExp()));
		return null;
	}
	
	@Override
	public Void visit(Create create, LGExtractVisitorMemory param) throws Throwable {
		String id = create.getId();
		LingObj o =objects.getByName(id); 
		if(o == null){
			throw new NotFoundLingObjException(id);
		}
	    param.getNamedActions().addAction(new CreateAction(o,new BooleanLeaf("true")));
		return null;
	}

	@Override
	public Void visit(Create_Conditional create_conditional, LGExtractVisitorMemory param) throws Throwable {
		String id = create_conditional.getId();
		LingObj o =objects.getByName(id); 
		if(o == null){
			throw new NotFoundLingObjException(id);
		}
	    create_conditional.getBoolexp().accept(this, param);
	    param.getNamedActions().addAction(new CreateAction(o,param.popBoolExp()));
		return null;
	}
	
	
	

	
	/* DEFINE */
	
	
	@Override
	public Void visit(Define_Star define_star, LGExtractVisitorMemory param) throws Throwable { 
		for(Define d : define_star.nodeList()){
			d.accept(this, param);			
		}			
		return null;
	}

	@Override
	public Void visit(Define define, LGExtractVisitorMemory param) throws Throwable {
		param.setType(define.getId());
		param.setName(define.getId2());		
		define.getLingobj().accept(this,param);
		objects.add(param.getLingObj());
		return null;
	}

	
	@Override
	public Void visit(Obj_Fs obj_fs, LGExtractVisitorMemory param) throws Throwable {
		FeatureStructure fs = new FeatureStructure();
		fs.setName(param.getName());
		fs.setType(param.getType());
		param.push(fs);
		param.setLingObj(fs);
		obj_fs.getFs().accept(this, param);
		param.pop();
		return null;
	}

	@Override
	public Void visit(Obj_List obj_list, LGExtractVisitorMemory param) throws Throwable {
		LingList list = new LingList(param.getType());
		list.setName(param.getName());
		param.setLingObj(list);
		param.push(list);
		obj_list.getList().accept(this, param);		
		param.pop();
		return null;
	}
	

	@Override
	public Void visit(Obj_Feature obj_feature, LGExtractVisitorMemory param) throws Throwable {
		StringFeature f = new StringFeature(obj_feature.getId(),obj_feature.getValue());
		f.setType(param.getType());
		f.setName(param.getName());
		param.setLingObj(f);
		objects.add(f);
		return null;
	}
	
	
	@Override
	public Void visit(Feature_Fs feature_fs, LGExtractVisitorMemory param) throws Throwable {
		String att = feature_fs.getId();
		FeatureStructure fs = new FeatureStructure(att);
		param.peek().addSimply(fs);
		param.push(fs);
		feature_fs.getSimplefs().accept(this,param);
		param.pop();
		return null;
	}

	
	
	@Override
	public Void visit(Feature_Fs_Inherits feature_fs_inherits, LGExtractVisitorMemory param) throws Throwable {
		FeatureStructure fs = new FeatureStructure();		
		String name = feature_fs_inherits.getId();
		LingObj o = objects.getByName(name);
		if(o == null){
			throw new NotFoundLingObjException(name);
		}
		if(!(o instanceof FeatureStructure)){
			throw new LGExtractException(name+" should be a feature structure !");
		}
		FeatureStructure parent = (FeatureStructure)o;
		fs.setType(parent.getType());
		fs.setParent(parent);
		param.peek().addSimply(fs);
		param.push(fs);
	    feature_fs_inherits.getSimplefs().accept(this, param);
		param.pop();		
		return null;
	}
	

	@Override
	public Void visit(Feature_List feature_list, LGExtractVisitorMemory param) throws Throwable {
		String att = feature_list.getId();
		LingList list = new LingList(att);
		param.peek().addSimply(list);		
		list.setType(att);
		param.push(list);
		feature_list.getList().accept(this, param);
		param.pop();
		return null;
	}

	@Override
	public Void visit(Feature_Value feature_value, LGExtractVisitorMemory param) throws Throwable {
		StringFeature f = new StringFeature(feature_value.getId(),feature_value.getValue());
		param.peek().addSimply(f);		
		return null;
	}

	@Override
	public Void visit(Features_Feature features_feature, LGExtractVisitorMemory param) throws Throwable {
		return features_feature.getFeature().accept(this, param);		
	}

	@Override
	public Void visit(Fs_Simple fs_simple, LGExtractVisitorMemory param) throws Throwable {		
		return fs_simple.getSimplefs().accept(this, param);
	}

	
	
	@Override
	public Void visit(Fs_Inherits fs_inherits, LGExtractVisitorMemory param) throws Throwable {
		String name = fs_inherits.getId();
		LingObj o = objects.getByName(name);
		if(o == null){
			throw new NotFoundLingObjException(name);
		}
		if(!(o instanceof FeatureStructure)){
			throw new LGExtractException(name+" should be a feature structure !");
		}
		FeatureStructure parent = (FeatureStructure)o;
		FeatureStructure current = (FeatureStructure)param.peek();
		current.setParent(parent);
		return fs_inherits.getSimplefs().accept(this, param);
	}


	@Override
	public Void visit(Feature_Name feature_name, LGExtractVisitorMemory param) throws Throwable {
		String name = feature_name.getId();
		LingObj o = objects.getByName(name);
		if(o == null){
			throw new NotFoundLingObjException(name);
		}
		param.peek().addSimply(o);	
		return null;
	}

	
	@Override
	public Void visit(List list, LGExtractVisitorMemory param) throws Throwable {		
		return list.getFeatures().accept(this, param);
	}

	@Override
	public Void visit(Empty_Simplefs empty_simplefs, LGExtractVisitorMemory param) throws Throwable {
		return null;
	}
	
	
	
	 @Override
		public Void visit(Empty_List empty_list, LGExtractVisitorMemory param) throws Throwable {
			return null;
		}
	
	

	@Override
	public Void visit(Several_Features several_features, LGExtractVisitorMemory param) throws Throwable {
		several_features.getFeature().accept(this, param);
		return several_features.getFeatures().accept(this, param);
	}

	@Override
	public Void visit(Simplefs simplefs, LGExtractVisitorMemory param) throws Throwable {		
		return simplefs.getFeatures().accept(this, param);
	}

	@Override
	public Void visit(Start start, LGExtractVisitorMemory param) throws Throwable {
		start.getDefine_star().accept(this, param);
		start.getProperty_star().accept(this,param);
		start.getElement_star().accept(this,param);
		return null;
	}

	public LGExtractActions getActions(){
		return actions;
	}
	
	public LGExtractLingObjects getObjects(){
		return objects;
	}
	
	public LGExtractFormatting getFormatting(){
		return formats;
	}

	@Override
	public Void visit(Propactionname_Always propactionname_always, LGExtractVisitorMemory param) throws Throwable {
		return null;
	}
	 
}
