/**
 * 
 *
 * Software: LGExtract
 * Authors: Matthieu Constant and Elsa Tolone
 *
 * Copyright (C) 2010-2011 Université Paris-Est Marne-la-Vallée
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 *
 */

package fr.umlv.lgextract;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.HashMap;

import jxl.read.biff.BiffException;

import org.jdom.Element;

import fr.umlv.lgextract.actions.LGExtractActions;
import fr.umlv.lgextract.exceptions.LGExtractException;
import fr.umlv.lgextract.exceptions.LGExtractNoEntryException;
import fr.umlv.lgextract.lobjects.LGExtractLingObjects;
import fr.umlv.lgextract.lobjects.LingObj;
import fr.umlv.lgextract.table.LGAbstractNormalTable;
import fr.umlv.lgextract.table.LGAbstractNormalTableEntry;
import fr.umlv.lgextract.table.LGAbstractTableOfTables;
import fr.umlv.lgextract.table.LGCsvTable;
import fr.umlv.lgextract.table.LGCsvTableOfTables;
import fr.umlv.lgextract.table.LGNormalTable;
import fr.umlv.lgextract.table.LGTableEntry;
import fr.umlv.lgextract.table.LGTableOfTables;

/**
 * Represents the process of generating the lexicon
 * 
 * @author Matthieu Constant
 *
 */

public class LGExtractProcess {
	private final LGExtractEnvironment env;
	private final LGExtractActions actions;
	private final LGExtractLingObjects objects;
	private final LGAbstractTableOfTables tableOfTables;
	private final HashMap<String, String> tableValues;
	private final HashMap<String, String> tableOfTableValues;
	private final Element root = new Element(LGExtract.XML_ROOT);
	LGExtractFormatting formatting;
	private LGExtractXMLWriter writer;
	private LGExtractTextualWriter textualWriter;
	
	
	public LGExtractProcess(LGExtractEnvironment env, LGExtractActions actions, LGExtractLingObjects objects,LGExtractFormatting formatting) throws IOException, IndexOutOfBoundsException, BiffException, LGExtractException{
		this.env = env;		
	
		this.tableValues = loadTableValues(env.getTableValuesFilename());
		this.tableOfTableValues = loadTableValues(env.getTableOfTableValuesFilename());
		if(env.getTableFormat().equals(LGExtractArguments.XLS_TABLE_FORMAT)){
			this.tableOfTables = new LGTableOfTables(env,tableOfTableValues);	
		}
		else{
			this.tableOfTables = new LGCsvTableOfTables(env,tableOfTableValues);
		}
		
		this.actions = actions;
		this.objects = objects;
		this.formatting = formatting;
		
		if(env.getOutput().equals(LGExtractArguments.XML_OUTPUT)){
			this.writer = new LGExtractXMLWriter(root,System.out);
		}
		if(env.getOutput().equals(LGExtractArguments.TXT_OUTPUT)){
			this.textualWriter = new LGExtractTextualWriter(new OutputStreamWriter(System.out));
		}
		if(this.writer == null && this.textualWriter == null){
			throw new LGExtractException("The output type is not well defined!");
		}
	}
	
	private HashMap<String,String> getDefaultTableValues(){
		HashMap<String, String> values = new HashMap<String, String>();
		values.put("+",LGExtract.TRUE_VALUE);
		values.put("-", LGExtract.FALSE_VALUE);
		values.put("O", LGExtract.FALSE_VALUE);
		values.put("?", LGExtract.FALSE_VALUE);
		values.put("?+", LGExtract.FALSE_VALUE);
		values.put("?-", LGExtract.FALSE_VALUE);
		values.put("x", LGExtract.FALSE_VALUE);
		values.put("X", LGExtract.FALSE_VALUE);
		values.put("~", LGExtract.FALSE_VALUE);
		values.put("/", LGExtract.FALSE_VALUE);
		values.put("//", LGExtract.FALSE_VALUE);
		return values;
	}
	
	private HashMap<String,String> loadTableValues(String filename){
		if(filename == null){			
			return getDefaultTableValues();
		}
		HashMap<String, String> values = new HashMap<String, String>();
		try{
		BufferedReader reader = new BufferedReader(new FileReader(filename));		
		
		String s;
		while(( s = reader.readLine()) != null){
			String[] tab = s.split(" ");
			if(tab.length == 2){
				values.put(tab[0], tab[1]);
			}
		}
		reader.close();
		}
		catch(IOException e){
			System.err.println("Warning: problem while reading "+ filename);
			System.err.println("Assigning default values...");
			return getDefaultTableValues(); 
		}
		return values;
	}
	
	
	private void tableProcess(String path,LGExtractEnvironment env,String name) throws IOException, LGExtractException, IndexOutOfBoundsException, BiffException{
		LGAbstractNormalTable table; 
		if(env.getTableFormat().equals(LGExtractArguments.XLS_TABLE_FORMAT)){
			table = new LGNormalTable(path,tableOfTables,tableValues,env.isDebug());
		}
		else{
			table = new LGCsvTable(path,tableOfTables,tableValues,env.isDebug());
		}
		env.incrementTableCount();
		System.err.println("************************* "+path);
		//System.err.println(table.getProperties());
		
		for(LGAbstractNormalTableEntry entry:table){
			try{
				ArrayList<LingObj> res = new ArrayList<LingObj>();
				//System.err.println(table.getName()+"****************ENTRY "+entry.getValue("<ENT>V")+" "+entry.getEntryId());
				LGExtractLingObjects clones = objects.copy();
				clones.resolve(entry);
				LGExtractActions actionClones = actions.clone(clones);
								
				for(String prop:tableOfTables.getProperties()){
					//System.err.println(entry.getEntryId()+"::"+prop+"=>"+entry.getValue(prop));
					/*if(!entry.getValue(prop).equals(LGExtract.FALSE_VALUE)){						 
						 System.err.println("=>"+prop+" "+entry.getValue(prop));
					 }*/
					//System.out.println(table+" "+prop+" "+table.getValue(prop));	
					actionClones.execute(prop,res, entry,entry.getValue(prop));
				}
								
				if(env.getOutput().equals(LGExtractArguments.XML_OUTPUT)){
					writer.writeEntry(res,formatting,entry);
				}
				if(env.getOutput().equals(LGExtractArguments.TXT_OUTPUT)){
					textualWriter.writeEntry(res, null,entry);
				}
				if(entry.isToEncode()){
					env.incrementToEncodeCount();
				}else{
					if(entry.isToComplete()){
						env.incrementToCompleteCount();
					}
				}
				env.incrementEntryCount();
			}
			catch(LGExtractNoEntryException e){
				env.incrementToCompleteCount();
				env.incrementToEncodeCount();
				env.incrementEntryCount();
				if(env.getOutput().equals(LGExtractArguments.XML_OUTPUT)){					
					writer.writeNoEntry(entry);
				}
				if(env.getOutput().equals(LGExtractArguments.TXT_OUTPUT)){
					textualWriter.writeNoEntry(entry);
				}				
			}
		}
	
		//System.err.println( table.getProperties());

	}

	private void propertyExistenceChecking() throws LGExtractException{
		for(String p:this.actions.getActionNames()){
			if(p.endsWith("__true")){
				//System.err.println("##"+p+"##");
				p = p.substring(1,p.length() - 7);
				//System.err.println("##"+p+"##");
			}
			else{
				//System.err.println("**"+p+"**");
				p = p.substring(1,p.length() - 8);
				//System.err.println("**"+p+"**");
			}
			if(!tableOfTables.getProperties().contains(p)){
				System.err.println("**"+p+"**");
				throw new LGExtractException(p+" does not exist in table of tables !");
			}
		}
		
	}
	
	public void go() throws LGExtractException,IOException,LGExtractNoEntryException, IndexOutOfBoundsException, BiffException{		
		propertyExistenceChecking();
		//int i = 0;
		for(LGTableEntry tableEntry:tableOfTables){
			try{
				//System.out.println(tableEntry.getEntryId());
				String tableName = tableEntry.getValue(LGExtract.TABLE_ID_PROPERTY);
				String suf = env.getTableFormat().equals(LGExtractArguments.XLS_TABLE_FORMAT)?"xls":"csv";
				tableProcess(env.getTableDirPath()+"/"+tableName+".lgt."+suf,env,tableName);				
			}
			catch(IOException e){
				System.err.println(e.toString());
			}
			/*if(i == 3){
				break;
			}
			i++;*/
		}		
		if(textualWriter != null) textualWriter.close();
		if(writer != null) writer.close();
	}
	
}
