/**
 * 
 *
 * Software: LGExtract
 * Authors: Matthieu Constant and Elsa Tolone
 *
 * Copyright (C) 2010-2011 Université Paris-Est Marne-la-Vallée
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 *
 */
package fr.umlv.lgextract.table;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jxl.Sheet;
import jxl.read.biff.BiffException;
import fr.umlv.lgextract.LGExtract;
import fr.umlv.lgextract.LGExtractEnvironment;

/**
 * @author Matthieu Constant
 *
 */
public class LGCsvTableOfTables extends LGAbstractTableOfTables{

	  final private HashMap<String,String> tableValues;
	  final private HashMap<String, Integer> tableIds = new HashMap<String, Integer>();
	  final private int startingCol;
	  final private int startingRow;
	  final private List<LGAbstractTableOfTablesEntry> entries = new ArrayList<LGAbstractTableOfTablesEntry>();
	 
	  
	  public LGCsvTableOfTables(LGExtractEnvironment env,HashMap<String,String> values) throws IndexOutOfBoundsException, IOException, BiffException{
	    	this.tableValues = values;
		    File file = new File(env.getTableOfTables());
		    if(!file.exists()){
		    	throw new IOException("Table of tables "+file+" does not exist!");
		    }
			this.startingCol = env.getHeaderColumn();
			this.startingRow = env.getHeaderRow();
			BufferedReader reader = new BufferedReader(
	                new InputStreamReader(
	                		 new FileInputStream(env.getTableOfTables()),"UTF-8"));
		    loadTableIds(reader);
		    loadEntriesAndProperties(reader);
		    reader.close();		    
	    }
	
	private String getCell(String cell){
		if(cell.isEmpty()){
			return cell;
		}
		return cell.substring(1,cell.length() -1);
	}
	  
	private void loadTableIds(BufferedReader reader) throws IOException{
		for(int i = 0 ; i < startingRow ; i++){
			reader.readLine();
		}
		String[] cells = reader.readLine().split(";");
		properties.put(LGExtract.TABLE_ID_PROPERTY, 0);
		for(int i = startingCol ; i < cells.length ; i++){
			int id = i - startingCol;
		   	tableIds.put(getCell(cells[i]), id);
		    LGAbstractTableOfTablesEntry e = new LGCsvTableOfTablesEntry(id, this);
		    e.addProp(getCell(cells[i]));
		    entries.add(e);
		}
		
	}
	  
	
	private void loadEntriesAndProperties(BufferedReader reader) throws IOException{
		String line;
		while((line = reader.readLine()) != null){
			String[] cells = line.split(";");
			if(cells.length == 1){
				continue;
			}
			properties.put(getCell(cells[startingCol]), properties.size());
			for(int i = 0 ; i < entries.size() ; i++){				
				LGAbstractTableOfTablesEntry e = entries.get(i);
				e.addProp(getCell(cells[i+startingCol]));
			}			
		}
	}
	
	public LGTableEntry getTableEntry(String tableName) {
		Integer propId = getPropertyId(LGExtract.TABLE_ID_PROPERTY);
		if(propId == null){
			return null;
		}
		/*System.out.println(propId);
		System.out.println(tableIds);
		System.out.println(tableName);*/
		return entries.get(tableIds.get(tableName));
	}
	
	
	
	@Override
	public String getIdProperty() {		
		return LGTableOfTables.ID_PROPERTY;
	}

	@Override
	public String getName() {		
		return null;
	}

	@Override
	public Integer getPropertyId(String property) {
		return properties.get(property);
	}

	@Override
	public Sheet getSheet() {
		throw new IllegalArgumentException("Sheet is not used in CVS table!!");
	}

	@Override
	public LGTableOfTables getTableOfTables() {
		return null; //no table of classes
	}

	@Override
	public Map<String, String> getValues() {
		return tableValues;
	}

	@Override
	public Iterator<LGAbstractTableOfTablesEntry> iterator() {
		return entries.iterator();
	}

}
