/**
 * 
 *
 * Software: LGExtract
 * Authors: Matthieu Constant and Elsa Tolone
 *
 * Copyright (C) 2010-2011 Université Paris-Est Marne-la-Vallée
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 *
 */

package fr.umlv.lgextract.lobjects;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.jdom.Element;

import fr.umlv.lgextract.LGExtractFormatting;
import fr.umlv.lgextract.exceptions.LGExtractException;
import fr.umlv.lgextract.exceptions.LGExtractNoEntryException;
import fr.umlv.lgextract.table.LGTableEntry;

/**
 * 
 * @author Matthieu Constant
 *
 */

public class FeatureStructure extends AbstractLingObj{

	private FeatureStructure parent;
	private HashMap<String, LingObj> map = new HashMap<String, LingObj>();
	

	
	public FeatureStructure(){
		
	}
	
	public FeatureStructure(String type){
		setType(type);
	}
	
	
	@Override
	public void resolve(LGTableEntry entry, LGExtractLingObjects objects) throws LGExtractException,LGExtractNoEntryException {
		for(LingObj o:map.values()){
			
			if(!objects.contains(o.getName())){
			     o.resolve(entry, objects);
			}
		}
	}

	@Override
	public List<LingObj> getObjects(String id) throws LGExtractException{
		LingObj o = this.get(id);
		if(o == null){
			throw new LGExtractException(id + " is not an attribute!");
		}
		ArrayList<LingObj> list = new ArrayList<LingObj>();
		list.add(o);
		return list;
	}

	public FeatureStructure getParent() {
		return parent;
	}

	public void setParent(FeatureStructure parent) {
		this.parent = parent;
	}

	@Override
	public String getType() {
		String type = super.getType();
		if(type == null){
			FeatureStructure parent = this.getParent();
			if(parent != null){				
				return parent.getType();
			}
			return null;
		}
		return type;
	}
	
	public LingObj get(String att){
		LingObj o = map.get(att);
		if(o == null){
			FeatureStructure parent = this.getParent();
			if(parent != null){				
				return parent.get(att);
			}
			return null;
		}
		return o;
	}
	
	public HashSet<String> getAttributes(){
		HashSet<String> hs = new HashSet<String>();
		if(parent != null){
			hs.addAll(parent.getAttributes());			
		}
		hs.addAll(map.keySet());
		return hs;
	}
	
	@Override
	public void add(LingObj o) throws LGExtractException {
		o.addIn(this);
	}	
	
	
	@Override
	public void addSimply(LingObj o) throws LGExtractException {
		map.put(o.getType(), o);
	}
	
	@Override
	public LingObj emptyInstance() {
		FeatureStructure fs = new FeatureStructure(getType());
		fs.setName(getName());
		return fs;
	}
	
	
	@Override
	public LingObj clone(LGExtractLingObjects objects) throws LGExtractException{
		String name = getName();
		FeatureStructure clone;
		if((name == null)||((clone= (FeatureStructure)objects.getByName(name)) == null)){
			clone = new FeatureStructure(getType());
			clone.setName(name);
		}
		if(getParent() != null){
		   clone.setParent((FeatureStructure)objects.getByName(getParent().getName()));
		}
		for(LingObj o:map.values()){
			clone.addSimply(o.clone(objects));
		}
		return clone;
	}
	
	
	public String toString(){
		String s = getType();
		Iterator<String> it = this.getAttributes().iterator();
		s += "=[";
		while(it.hasNext()){
			s += this.get(it.next()).toString();			
			if(it.hasNext()){
				s += ","; 
			}			
		}
		s += "]";
		return s;
	}
	
	public Collection<LingObj> getAll(){
		Set<LingObj> set = new HashSet<LingObj>();
		set.addAll(map.values());
		if(parent != null){
			set.addAll(parent.getAll());
		}
		return set;
	}
	
	@Override
	public void toXML(Element parent, LGExtractFormatting formatting) {
		Element par = parent;
		if(formatting.isElement(getType())){
			par = new Element(formatting.getElement(getType()));
			parent.addContent(par);
		}
		
		for(String att:this.getAttributes()){
			get(att).toXML(par, formatting);
		}
	}
	
	public HashMap<String, LingObj> getMap() {
		return map;
	}	
	
}
