/**
 * 
 *
 * Software: LGExtract
 * Authors: Matthieu Constant and Elsa Tolone
 *
 * Copyright (C) 2010-2011 Université Paris-Est Marne-la-Vallée
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 *
 */

package fr.umlv.lgextract;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;

/**
 * Represents command line arguments parsed with JSAP
 * 
 * 
 * @author Matthieu Constant
 *
 */

public class LGExtractArguments {
	final private static String SCRIPT_OPT = "script";
	final private static String TABLE_DIR_PATH_OPT = "tableDirPath";
	final private static String HEADER_COLUMN = "headerColumn";
	final private static String HEADER_ROW = "headerRow";
	final private static String DEBUG_OPT = "debug";
	final private static String TDT_OPT = "tdt";
	final private static String TDT_VALUES_OPT = "tdtValues";
	final private static String TABLE_VALUES_OPT = "values";
	final private static String OUTPUT_OPT = "output";
	final public static String TXT_OUTPUT = "txt";
	final public static String XML_OUTPUT = "xml";
	final public static String TABLE_FORMAT = "tableFormat";
	final public static String XLS_TABLE_FORMAT = "xls";
	final public static String CSV_TABLE_FORMAT = "csv";
	
	final private String tableOfTables;
	final private int headerRow;
	final private int headerColumn;
	final private String scriptFilename;
	final private String tableDirPath;
	final private String tableValuesFilename;
	final private String tableOfTableValuesFilename;
	final private boolean debug;
	final private String output;
	final private String tableFormat;
	
	public LGExtractArguments(String[] args) throws JSAPException{
		JSAP jsap;
		jsap = createJSAP();

		JSAPResult config = jsap.parse(args);
		if(!config.success()){
			System.err.print("Usage: ");
			System.err.print("java "+LGExtract.class.getCanonicalName()+" ");
            System.err.println(jsap.getUsage());
            System.err.print(jsap.getHelp());
            System.exit(1);
		}
		tableOfTables = config.getString(TDT_OPT);
		headerRow = config.getInt(HEADER_ROW);
		headerColumn = config.getInt(HEADER_COLUMN);
		tableDirPath = config.getString(TABLE_DIR_PATH_OPT);
		scriptFilename = config.getString(SCRIPT_OPT);
		tableValuesFilename = config.getString(TABLE_VALUES_OPT);
		tableOfTableValuesFilename = config.getString(TDT_VALUES_OPT);
		debug = config.getBoolean(DEBUG_OPT);
		output = config.getString(OUTPUT_OPT);
		tableFormat = config.getString(TABLE_FORMAT);
	}
	
	
	private static JSAP createJSAP() throws JSAPException{
		JSAP jsap = new JSAP();
		FlaggedOption opt = new FlaggedOption(SCRIPT_OPT)
        .setStringParser(JSAP.STRING_PARSER)
        .setRequired(true)
        .setShortFlag(JSAP.NO_SHORTFLAG)
        .setLongFlag(SCRIPT_OPT)        
        ;
		opt.setHelp("Script pathname (the script file must be encoded in UTF-8)");
		jsap.registerParameter(opt);
		
		opt = new FlaggedOption(TABLE_DIR_PATH_OPT)
        .setStringParser(JSAP.STRING_PARSER)
        .setRequired(true)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(TABLE_DIR_PATH_OPT);
		opt.setHelp("Path of the directory containing the tables.");
		jsap.registerParameter(opt);
		
		opt = new FlaggedOption(HEADER_COLUMN)
        .setStringParser(JSAP.INTEGER_PARSER)
        .setRequired(true)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(HEADER_COLUMN);
		opt.setHelp("Column id of the property headers in the table of classes (first column is 0)");
		jsap.registerParameter(opt);
		
		opt = new FlaggedOption(HEADER_ROW)
        .setStringParser(JSAP.INTEGER_PARSER)
        .setRequired(true)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(HEADER_ROW);
		opt.setHelp("Row id of the table headers in the table of classes (first row is 0)");
		jsap.registerParameter(opt);
		
		Switch sw = new Switch(DEBUG_OPT).setLongFlag(DEBUG_OPT).setShortFlag(JSAP.NO_SHORTFLAG);
		sw.setHelp("indicates that the program is run in debug mode (if not present, it is run in the normal mode)");
		jsap.registerParameter(sw);
		
		
		opt = new FlaggedOption(TDT_VALUES_OPT)
        .setStringParser(JSAP.STRING_PARSER)
        .setRequired(false)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(TDT_VALUES_OPT);
		opt.setHelp("Path of the file containing interpretations of the values in the table of classes. Default: standard lexicon-grammar value interpretation.");
		jsap.registerParameter(opt);
		
		opt = new FlaggedOption(TABLE_VALUES_OPT)
        .setStringParser(JSAP.STRING_PARSER)
        .setRequired(false)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(TABLE_VALUES_OPT);
		opt.setHelp("Path of the file containing interpretations of the values in the lexicon-grammar tables. Default: standard lexicon-grammar value interpretation.");
		jsap.registerParameter(opt);
		
		opt = new FlaggedOption(OUTPUT_OPT)
        .setStringParser(JSAP.STRING_PARSER)
        .setRequired(false)
        .setDefault(LGExtractArguments.TXT_OUTPUT)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(OUTPUT_OPT);
		opt.setHelp("Output types: textual ("+TXT_OUTPUT+") or xml ("+XML_OUTPUT+"). Default value: "+TXT_OUTPUT);
		jsap.registerParameter(opt);
		
		opt = new FlaggedOption(TABLE_FORMAT)
        .setStringParser(JSAP.STRING_PARSER)
        .setRequired(false)
        .setDefault(LGExtractArguments.TXT_OUTPUT)
        .setShortFlag(JSAP.NO_SHORTFLAG)
		.setLongFlag(TABLE_FORMAT);
		opt.setHelp("table format: excel ("+XLS_TABLE_FORMAT+") or csv ("+CSV_TABLE_FORMAT+"). Default value: "+CSV_TABLE_FORMAT);
		jsap.registerParameter(opt);		
		
		
		UnflaggedOption opt1 = new UnflaggedOption(TDT_OPT)
        .setStringParser(JSAP.STRING_PARSER)
        .setGreedy(false)
        .setRequired(true);
		opt1.setHelp("List of the tables of classes used!");
		jsap.registerParameter(opt1);
		
		
		return jsap;
	}


	/**
	 * @return the tablesOfTables
	 */
	public String getTableOfTables() {
		return tableOfTables;
	}


	/**
	 * @return the headerRow
	 */
	public int getHeaderRow() {
		return headerRow;
	}


	/**
	 * @return the headerColumn
	 */
	public int getHeaderColumn() {
		return headerColumn;
	}


	/**
	 * @return the scriptFilename
	 */
	public String getScriptFilename() {
		return scriptFilename;
	}


	/**
	 * @return the tableDirPath
	 */
	public String getTableDirPath() {
		return tableDirPath;
	}


	/**
	 * @return the tableValuesFilename
	 */
	public String getTableValuesFilename() {
		return tableValuesFilename;
	}


	/**
	 * @return the tableOfTableValuesFilename
	 */
	public String getTableOfTableValuesFilename() {
		return tableOfTableValuesFilename;
	}


	/**
	 * @return the debug
	 */
	public boolean isDebug() {
		return debug;
	}


	/**
	 * @return the output
	 */
	public String getOutput() {
		return output;
	}


	/**
	 * @return the tableFormat
	 */
	public String getTableFormat() {
		return tableFormat;
	}
	
  
	
}
