#include "XmlMorphoDictionaryEntryFormatter.h"

using namespace std;
using namespace LinguisticDefinition;
using namespace MorphologyDefinition;

/**
 *
 */
XmlMorphoDictionaryEntryFormatter::
XmlMorphoDictionaryEntryFormatter(const LingDef &lingDef) :
  MorphoDictionaryEntryFormatter(lingDef),
  d_featuresFormatter(lingDef) {
}

/**
 *
 */
MorphoDictionaryEntry
XmlMorphoDictionaryEntryFormatter::
createMorphoDictionaryEntry(const string &xmlData) {
  MorphoDictionaryEntry entry(getLingDef());
  populateMorphoDictionaryEntry(xmlData, entry);
  return entry;
}

/**
 *
 */
MorphoDictionaryEntry
XmlMorphoDictionaryEntryFormatter::
createMorphoDictionaryEntry(xmlNodePtr entryNode) {
  MorphoDictionaryEntry entry(getLingDef());
  populateMorphoDictionaryEntry(entryNode, entry);
  return entry;
}

/**
 *
 */
bool
XmlMorphoDictionaryEntryFormatter::
populateMorphoDictionaryEntry(const string &xmlData,
			      MorphoDictionaryEntry &entry) {
  xmlDocPtr doc = xmlParseDoc((xmlChar *) xmlData.c_str());
  if (doc != NULL) {
    bool populationOk = populateMorphoDictionaryEntry(doc, entry);
    xmlFreeDoc(doc);
    return populationOk;
  }
  return false;
}

/**
 *
 */
bool
XmlMorphoDictionaryEntryFormatter::
populateMorphoDictionaryEntry(xmlDocPtr doc,
			      MorphoDictionaryEntry &entry) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(doc);
  xmlXPathObjectPtr xpObj =
    xmlXPathEvalExpression((xmlChar *) "//Entry", xpContext);
  if (xpObj != NULL) {
    xmlNodeSetPtr nodeSet = xpObj->nodesetval;
    if (nodeSet != NULL && nodeSet->nodeNr > 0) {
      populateMorphoDictionaryEntry(nodeSet->nodeTab[0], entry);
    }
    xmlXPathFreeObject(xpObj);
  }
    
  xmlXPathFreeContext(xpContext);

  return true;
}

/**
 *
 */
bool
XmlMorphoDictionaryEntryFormatter::
populateMorphoDictionaryEntry(xmlNodePtr entryNode,
			      MorphoDictionaryEntry &entry) {
  bool populationOk = true;

  xmlXPathContextPtr xpContext = xmlXPathNewContext(entryNode->doc);
  xpContext->node = entryNode;

  {
    xmlChar *tmp = xmlGetProp(entryNode, (xmlChar *) "pos");
    if (tmp != NULL) {
      entry.setPosName((char *) tmp);
      xmlFree(tmp);
    }
  }

  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Lemma",
					   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL && nodeSet->nodeNr > 0) {
	xmlChar *tmp = xmlNodeGetContent(nodeSet->nodeTab[0]);
	if (tmp != NULL) {
	  entry.setLemma((char *) tmp);
	  xmlFree(tmp);
	}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Feature",
					 xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr featureNode = nodeSet->nodeTab[i];

	  bool isConstraint = false;
	  {
	    xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "constraint");
	    if (tmp != NULL) {
	      isConstraint = xmlStrcasecmp(tmp, (xmlChar *) "false") != 0;
	      xmlFree(tmp);
	    }
	  }

	  {
	    xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "name");
	    if (tmp != NULL) {
	      entry.addFeature((char *) tmp, isConstraint);
	      xmlFree(tmp);
	    }
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Inflection",
					   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr inflectionNode = nodeSet->nodeTab[i];

	  string classId;
	  {
	    xmlChar *tmp = xmlGetProp(inflectionNode,
				      (xmlChar *) "classId");
	    if (tmp != NULL) {
	      classId = (char *) tmp;
	      xmlFree(tmp);
	    }
	  }

	  string declinationCode;
	  {
	    xmlChar *tmp = xmlGetProp(inflectionNode,
				      (xmlChar *) "declinationCode");
	    if (tmp != NULL) {
	      declinationCode = (char *) tmp;
	      xmlFree(tmp);
	    }
	  }

	  MorphoDictionaryEntry::Inflection &inflection =
	    entry.createInflection(classId, declinationCode);

	  {
	    xmlXPathContextPtr xpContext2 =
	      xmlXPathNewContext(inflectionNode->doc);
	    xpContext2->node = inflectionNode;

	    xmlXPathObjectPtr xpObj2 = xmlXPathEval((xmlChar *) "Stem",
						    xpContext2);
	    if (xpObj2 != NULL) {
	      xmlNodeSetPtr nodeSet2 = xpObj2->nodesetval;
	      if (nodeSet2 != NULL && nodeSet2->nodeNr > 0) {
		xmlChar *tmp = xmlNodeGetContent(nodeSet2->nodeTab[0]);
		if (tmp != NULL) {
		  inflection.setStem((char *) tmp);
		  xmlFree(tmp);
		}
	      }
	      xmlXPathFreeObject(xpObj2);
	    }

	    xpObj2 = xmlXPathEval((xmlChar *) "Feature", xpContext2);
	    if (xpObj2 != NULL) {
	      xmlNodeSetPtr nodeSet2 = xpObj2->nodesetval;
	      if (nodeSet2 != NULL) {
		{for (int j = 0; j < nodeSet2->nodeNr; j++) {
		  xmlNodePtr featureNode = nodeSet2->nodeTab[j];

		  bool isConstraint = false;
		  {
		    xmlChar *tmp = xmlGetProp(featureNode,
					      (xmlChar *) "constraint");
		    if (tmp != NULL) {
		      isConstraint =
			xmlStrcasecmp(tmp, (xmlChar *) "false") != 0;
		      xmlFree(tmp);
		    }
		  }

		  {
		    xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "name");
		    if (tmp != NULL) {
		      inflection.addFeature((char *) tmp, isConstraint);
		      xmlFree(tmp);
		    }
		  }
		}}
	      }
	      xmlXPathFreeObject(xpObj2);
	    }

	    xmlXPathFreeContext(xpContext2);
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Derivation",
					   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr derivationNode = nodeSet->nodeTab[i];

	  string classId;
	  {
	    xmlChar *tmp = xmlGetProp(derivationNode,
				      (xmlChar *) "classId");
	    if (tmp != NULL) {
	      classId = (char *) tmp;
	      xmlFree(tmp);
	    }
	  }

	  string declinationCode;
	  {
	    xmlChar *tmp = xmlGetProp(derivationNode,
				      (xmlChar *) "declinationCode");
	    if (tmp != NULL) {
	      declinationCode = (char *) tmp;
	      xmlFree(tmp);
	    }
	  }

	  string posName;
	  {
	    xmlChar *tmp = xmlGetProp(derivationNode,
				      (xmlChar *) "pos");
	    if (tmp != NULL) {
	      posName = (char *) tmp;
	      xmlFree(tmp);
	    }
	  }

	  entry.createDerivation(posName, classId, declinationCode);
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);

  return populationOk;
}

/**
 *
 */
string
XmlMorphoDictionaryEntryFormatter::
output(const MorphoDictionaryEntry &entry) const {
  stringstream s;
  output(entry, s);
  return s.str();
}

/**
 *
 */
void
XmlMorphoDictionaryEntryFormatter::output(const MorphoDictionaryEntry &entry,
					  ostream &out) const {
  xmlOutputBufferPtr b = xmlAllocOutputBuffer(NULL);
  xmlTextWriterPtr writer = xmlNewTextWriter(b);
  xmlTextWriterSetIndent(writer, 1);
  output(entry, writer);
  out << xmlBufferContent(b->buffer);
  xmlFreeTextWriter(writer);
}

/**
 *
 */
void
XmlMorphoDictionaryEntryFormatter::output(const MorphoDictionaryEntry &entry,
					  xmlTextWriterPtr writer) const {

  xmlTextWriterStartElement(writer, (xmlChar *) "Entry");
  xmlTextWriterWriteAttribute(writer,
			      (xmlChar *) "pos",
			      (xmlChar *) entry.getPosName().c_str());


  xmlTextWriterStartElement(writer, (xmlChar *) "Lemma");
  xmlTextWriterWriteString(writer, (xmlChar *) entry.getLemma().c_str());
  xmlTextWriterEndElement(writer); //Lemma

  {
    set<string> featureConstraints;
    {for (MorphoDictionaryEntry::FeatureIterator
	    it = entry.featureConstraintsBegin();
	  it != entry.featureConstraintsEnd(); ++it) {
      featureConstraints.insert(*it);
    }}

    //TODO: handle positive/negative?
    {for (MorphoDictionaryEntry::FeatureIterator
	    it = entry.featuresBegin();
	  it != entry.featuresEnd(); ++it) {
      xmlTextWriterStartElement(writer, (xmlChar *) "Feature");
      xmlTextWriterWriteAttribute(writer,
				  (xmlChar *) "name",
				  (xmlChar *) (*it).c_str());

      if (featureConstraints.find(*it) != featureConstraints.end()) {
	xmlTextWriterWriteAttribute(writer,
				    (xmlChar *) "constraint",
				    (xmlChar *) "true");
      }

      xmlTextWriterEndElement(writer); //Feature
    }}
  }

  {for (MorphoDictionaryEntry::InflectionIterator
	  it = entry.inflectionsBegin();
	it != entry.inflectionsEnd(); ++it) {
    const MorphoDictionaryEntry::Inflection *inflection = *it;
    
    if (inflection->isDerivation()) {
      xmlTextWriterStartElement(writer, (xmlChar *) "Derivation");
    } else {
      xmlTextWriterStartElement(writer, (xmlChar *) "Inflection");
    }

    if (inflection->getClassId() != "") {
      xmlTextWriterWriteAttribute(writer,
				  (xmlChar *) "classId",
				  (xmlChar *) inflection->getClassId().
				  c_str());

      if (inflection->getDeclinationCode() != "") {
	xmlTextWriterWriteAttribute(writer,
				    (xmlChar *) "declinationCode",
				    (xmlChar *) inflection->
				    getDeclinationCode().
				    c_str());
      }
    }
    
    if (inflection->isDerivation()) {
      string posName = inflection->getPosName();
      if (posName == "") {
	const LingFeatures *features = inflection->getFeatures();
	if (features != NULL) {
	  const LingDef::Pos *posDef = features->getPosDef();
	  if (posDef != NULL) {
	    posName = posDef->getName();
	  }
	}
      }
      xmlTextWriterWriteAttribute(writer,
				  (xmlChar *) "pos",
				  (xmlChar *) posName.c_str());
    }

    if (inflection->getStem() != "" &&
	inflection->getStem() != entry.getLemma()) {
      xmlTextWriterStartElement(writer, (xmlChar *) "Stem");
      xmlTextWriterWriteString(writer,
			       (xmlChar *) inflection->getStem().c_str());
      xmlTextWriterEndElement(writer); //Stem
    }

    {
      set<string> featureConstraints;
      {for (MorphoDictionaryEntry::FeatureIterator
	      it = inflection->featureConstraintsBegin();
	    it != inflection->featureConstraintsEnd(); ++it) {
	featureConstraints.insert(*it);
      }}

      //TODO: handle positive/negative?
      {for (MorphoDictionaryEntry::FeatureIterator
	      it = inflection->featuresBegin();
	    it != inflection->featuresEnd(); ++it) {
	xmlTextWriterStartElement(writer, (xmlChar *) "Feature");
	xmlTextWriterWriteAttribute(writer,
				    (xmlChar *) "name",
				    (xmlChar *) (*it).c_str());

	if (featureConstraints.find(*it) != featureConstraints.end()) {
	  xmlTextWriterWriteAttribute(writer,
				      (xmlChar *) "constraint",
				      (xmlChar *) "true");
	}

	xmlTextWriterEndElement(writer); //Feature
      }}
    }

    xmlTextWriterEndElement(writer); //Inflection
  }}

  {
    bool firstInflected = true;


    {for (MorphoDictionaryEntry::InflectionIterator
	    it = entry.inflectionsBegin();
	  it != entry.inflectionsEnd(); ++it) {
      const MorphoDictionaryEntry::Inflection *inflection = *it;
   
      {for (MorphoDictionaryEntry::Inflection::InflectedIterator
	      it2 = inflection->inflectedBegin();
	    it2 != inflection->inflectedEnd(); ++it2) {
	if (firstInflected) {
	  xmlTextWriterStartElement(writer, (xmlChar *) "Inflected");
	  firstInflected = false;
	}

	d_featuresFormatter.output(**it2, writer);
      }}

    }}

    if (!firstInflected) {
      xmlTextWriterEndElement(writer); //Inflected
    }
  }

  xmlTextWriterEndElement(writer); //Entry
}
