#include "MorphoDictionaryEntry.h"

using namespace std;
using namespace LinguisticDefinition;
using namespace MorphologyDefinition;

/**
 *
 */
MorphoDictionaryEntry::
MorphoDictionaryEntry(const LingDef &lingDef) :
  d_lingDef(lingDef) {
}

/**
 *
 */
MorphoDictionaryEntry::
MorphoDictionaryEntry(const LingDef &lingDef,
		      const string &posName,
		      const string &lemma) :
  d_lingDef(lingDef),
  d_posName(posName),
  d_lemma(lemma) {
}

/**
 *
 */
MorphoDictionaryEntry::
MorphoDictionaryEntry(const MorphoDictionaryEntry &o) :
  d_lingDef(o.d_lingDef),
  d_posName(o.d_posName),
  d_lemma(o.d_lemma),
  d_features(o.d_features) {
}

/**
 *
 */
void MorphoDictionaryEntry::setPosName(const string &val) {
  d_posName = val;
}

/**
 *
 */
void MorphoDictionaryEntry::setLemma(const string &val) {
  d_lemma = val;
}

/**
 *
 */
const string &MorphoDictionaryEntry::getPosName() const {
  return d_posName;
}

/**
 *
 */
const string &MorphoDictionaryEntry::getLemma() const {
  return d_lemma;
}

/**
 *
 */
MorphoDictionaryEntry::~MorphoDictionaryEntry() {
  {for (InflectionIterator it = d_inflections.begin();
	it != d_inflections.end(); ++it) {
    delete *it;
  }}
}

/**
 *
 */
void MorphoDictionaryEntry::addFeature(const string &featureName,
				       bool isConstraint) {
  d_features.push_back(featureName);
  if (isConstraint) {
    d_featureConstraints.push_back(featureName);
  }
}

/**
 *
 */
void MorphoDictionaryEntry::Inflection::addFeature(const string &featureName,
						   bool isConstraint) {
  d_features.push_back(featureName);
  if (isConstraint) {
    d_featureConstraints.push_back(featureName);
  }

  if (d_featuresObj != NULL) {
    d_featuresObj->set(featureName);
  }
}

/**
 *
 */
void MorphoDictionaryEntry::inflect(const Inflector &inflector) {
  {for (InflectionIterator it = d_inflections.begin();
	it != d_inflections.end(); ++it) {
    (*it)->inflect(inflector);
  }}
}

/**
 *
 */
void MorphoDictionaryEntry::clearInflected() {
  {for (InflectionIterator it = d_inflections.begin();
	it != d_inflections.end(); ++it) {
    (*it)->clearInflected();
  }}
}

/**
 *
 */
void MorphoDictionaryEntry::Inflection::clearInflected() {
  {for (InflectedIterator it = d_inflected.begin();
	it != d_inflected.end(); ++it) {
    delete *it;
  }}
  d_inflected.clear();
}

/**
 *
 */
void
MorphoDictionaryEntry::Inflection::
inflectSub(const Inflector::Inflections &inflections,
	   const LinguisticDefinition::LingFeatures &defaultFeatures) {
  {for (Inflector::Inflections::InflectionIterator
	  it = inflections.inflectionsBegin();
	it != inflections.inflectionsEnd(); ++it) {
    const Inflector::Inflection *inflection = *it;

    LingFeatures *inflected = new LingFeatures(defaultFeatures);

    inflected->setForm(inflection->getForm());

    {for (Inflector::Inflection::FeatureIterator
	    it2 = inflection->featuresBegin();
	  it2 != inflection->featuresEnd(); ++it2) {
      const MorphoDef::Feature *featureDef = *it2;
      inflected->set(featureDef->getName());
    }}

    // Restrictions by features in entry
    bool restrictionsOk = true;

    {for (MorphoDictionaryEntry::FeatureIterator
	    it = d_parent.featureConstraintsBegin();
	  restrictionsOk && it != d_parent.featureConstraintsEnd(); ++it) {
      if (!inflected->has(*it)) {
	restrictionsOk = false;
      }
    }}

    if (restrictionsOk) {
      {for (MorphoDictionaryEntry::FeatureIterator
	      it = featureConstraintsBegin();
	    restrictionsOk && it != featureConstraintsEnd(); ++it) {
	if (!inflected->has(*it)) {
	  restrictionsOk = false;
	}
      }}
    }

    if (restrictionsOk) {
      d_inflected.push_back(inflected);
    } else {
      delete inflected;
    }
  }}

  {for (Inflector::Inflections::DerivationIterator
	  it = inflections.derivationsBegin();
	it != inflections.derivationsEnd(); ++it) {
    const Inflector::Derivation *derivation = *it;

    const string derivationPosName = derivation->getPosName();

    if (derivationPosName != "") {
      const LingDef::Pos *derivationPosDef =
	defaultFeatures.getLingDef()->getPos(derivationPosName);
      if (derivationPosDef == NULL) {
	//ERROR
      } else {
	const Inflector::Inflections &derivationInflections =
	  (*it)->getInflections();

	LingFeatures derivationDefaultFeatures(*derivationPosDef);
  	derivationDefaultFeatures.setLemma(derivationInflections.getLemma());

	inflectSub(derivationInflections,
		   derivationDefaultFeatures);
      }
    } else {
      inflectSub((*it)->getInflections(), defaultFeatures);
    }

  }}
}

/**
 *
 */
const LingFeatures *MorphoDictionaryEntry::Inflection::getFeatures() const {
  return d_featuresObj;
}

/**
 *
 */
const string &MorphoDictionaryEntry::Inflection::getClassId() const {
  return d_classId;
}

/**
 *
 */
const string &MorphoDictionaryEntry::Inflection::getPosName() const {
  return d_posName;
}

/**
 *
 */
const string &MorphoDictionaryEntry::Inflection::getStem() const {
  return d_stem;
}

/**
 *
 */
void MorphoDictionaryEntry::Inflection::setStem(const string &val) {
  d_stem = val;
}

/**
 *
 */
const string &MorphoDictionaryEntry::Inflection::getDeclinationCode()
  const {
  return d_declinationCode;
}

/**
 *
 */
bool MorphoDictionaryEntry::Inflection::isDerivation() const {
  return d_isDerivation;
}

/**
 *
 */
void MorphoDictionaryEntry::Inflection::inflect(const Inflector &inflector) {

  if (d_featuresObj != NULL) {

    const MorphoDef::Class *classDef =
      inflector.getMorphoDef().getClass(d_classId);

    if (classDef != NULL) {
      const vector<const MorphoDef::Declination *> *declinationDefs =
	classDef->getDeclinations(d_declinationCode);

      if (declinationDefs != NULL) {
	string lemma = d_stem;
	if (lemma == "") {
	  lemma = d_featuresObj->getLemma();
	}

	bool triggered = false;
	{for (vector<const MorphoDef::Declination *>::const_iterator
		it = declinationDefs->begin();
	      it != declinationDefs->end() && !triggered; ++it) {
	  Inflector::Inflections inflections(inflector, lemma,
					     **it);
	  triggered = inflector.inflect(inflections);
	
	  if (triggered) {
	    inflectSub(inflections, *d_featuresObj);
	    break;
	  }
	}}
      }
    } else {
      // Do not call inflector, simply use the stem as form
      LingFeatures *inflected = new LingFeatures(*d_featuresObj);
      if (d_stem != "") {
	inflected->setForm(d_stem);
      } else {
	inflected->setForm(inflected->getLemma());
      }
      d_inflected.push_back(inflected);
    }
  }

}

/**
 *
 */
MorphoDictionaryEntry::Inflection::InflectedIterator
MorphoDictionaryEntry::Inflection::inflectedBegin() const {
  return d_inflected.begin();
}

/**
 *
 */
MorphoDictionaryEntry::Inflection::InflectedIterator
MorphoDictionaryEntry::Inflection::inflectedEnd() const {
  return d_inflected.end();
}

/**
 *
 */
MorphoDictionaryEntry::Inflection::Inflection(MorphoDictionaryEntry &parent) :
  d_parent(parent),
  d_isDerivation(false) {
  init(parent, parent.getPosName(), "", "");
}

/**
 *
 */
MorphoDictionaryEntry::Inflection::Inflection(MorphoDictionaryEntry &parent,
					      const string &classId,
					      const string &declCode) :
  d_parent(parent),
  d_isDerivation(false) {
  init(parent, parent.getPosName(), classId, declCode);
}

/**
 *
 */
MorphoDictionaryEntry::Inflection::Inflection(MorphoDictionaryEntry &parent,
					      const string &posName,
					      const string &classId,
					      const string &declCode) :
  d_parent(parent),
  d_isDerivation(true) {
  init(parent, posName, classId, declCode);
}

/**
 *
 */
void MorphoDictionaryEntry::Inflection::init(MorphoDictionaryEntry &parent,
					     const string &posName,
					     const string &classId,
					     const string &declCode) {
  d_posName = posName;
  d_classId = classId;
  d_declinationCode = declCode;

  const LingDef::Pos *posDef = parent.getLingDef().getPos(posName);

  if (posDef == NULL) {
    //ERROR
    d_featuresObj = NULL;
  } else {
    d_featuresObj = new LingFeatures(*posDef);
    d_featuresObj->setLemma(parent.getLemma());

    {for (MorphoDictionaryEntry::FeatureIterator it = parent.featuresBegin();
	  it != parent.featuresEnd(); ++it) {
      d_featuresObj->set(*it);
    }}
  }
}

/**
 *
 */
MorphoDictionaryEntry::Inflection::~Inflection() {
  delete d_featuresObj;
  {for (InflectedIterator it = d_inflected.begin();
	it != d_inflected.end(); ++it) {
    delete *it;
  }}
}

/**
 *
 */
MorphoDictionaryEntry::Inflection::Inflection(const Inflection &o) :
  d_parent(o.d_parent),
  d_isDerivation(o.d_isDerivation),
  d_classId(o.d_classId),
  d_declinationCode(o.d_declinationCode),
  d_features(o.d_features),
  d_featureConstraints(o.d_featureConstraints),
  d_featuresObj(o.d_featuresObj == NULL ?
		NULL : new LingFeatures(*o.d_featuresObj)) {

  {for (InflectedIterator it = o.d_inflected.begin();
	it != o.d_inflected.end(); ++it) {
    d_inflected.push_back(new LingFeatures(**it));
  }}
}

/**
 *
 */
MorphoDictionaryEntry::Inflection &
MorphoDictionaryEntry::createInflection() {
  Inflection *newInflection = new Inflection(*this);
  d_inflections.push_back(newInflection);
  return *newInflection;
}


/**
 *
 */
MorphoDictionaryEntry::Inflection &
MorphoDictionaryEntry::createInflection(const string &classId,
					const string &declinationCode) {
  Inflection *newInflection = new Inflection(*this, classId, declinationCode);
  d_inflections.push_back(newInflection);
  return *newInflection;
}

/**
 *
 */
MorphoDictionaryEntry::Inflection &
MorphoDictionaryEntry::createDerivation(const string &pos,
					const string &classId,
					const string &declinationCode) {
  Inflection *newInflection = new Inflection(*this,
					     pos, classId, declinationCode);
  d_inflections.push_back(newInflection);
  return *newInflection;
}

/**
 *
 */
const LingDef &MorphoDictionaryEntry::getLingDef() const {
  return d_lingDef;
}

/**
 *
 */
MorphoDictionaryEntry::InflectionIterator
MorphoDictionaryEntry::inflectionsBegin() const {
  return d_inflections.begin();
}

/**
 *
 */
MorphoDictionaryEntry::InflectionIterator
MorphoDictionaryEntry::inflectionsEnd() const {
  return d_inflections.end();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::featuresBegin() const {
  return d_features.begin();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::featuresEnd() const {
  return d_features.end();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::featureConstraintsBegin() const {
  return d_featureConstraints.begin();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::featureConstraintsEnd() const {
  return d_featureConstraints.end();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::Inflection::featuresBegin() const {
  return d_features.begin();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::Inflection::featuresEnd() const {
  return d_features.end();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::Inflection::featureConstraintsBegin() const {
  return d_featureConstraints.begin();
}

/**
 *
 */
MorphoDictionaryEntry::FeatureIterator
MorphoDictionaryEntry::Inflection::featureConstraintsEnd() const {
  return d_featureConstraints.end();
}
