#include "MorphologyDefinition/XmlMorphoDefFormatter.h"

using namespace std;
using namespace MorphologyDefinition;

/**
 *
 */
XmlMorphoDefFormatter::XmlMorphoDefFormatter() {
}

/**
 *
 */
MorphoDef
XmlMorphoDefFormatter::
createMorphoDef(const string &xmlData,
                           const string &isoLanguageCode) {
  MorphoDef morphoDef(isoLanguageCode);
  xmlDocPtr doc = xmlParseDoc((xmlChar *) xmlData.c_str());
  if (doc != NULL) {
    populateMorphoDef(doc, morphoDef);
    xmlFreeDoc(doc);
  }
  return morphoDef;
}

/**
 *
 */
MorphoDef
XmlMorphoDefFormatter::
createMorphoDef(xmlDocPtr doc,
                           const string &isoLanguageCode) {
  MorphoDef morphoDef(isoLanguageCode);
  populateMorphoDef(doc, morphoDef);
  return morphoDef;
}

/**
 *
 */
void
XmlMorphoDefFormatter::
populateMorphoDef(const string &xmlData,
                             MorphoDef &morphoDef) {
  xmlDocPtr doc = xmlParseDoc((xmlChar *) xmlData.c_str());
  if (doc != NULL) {
    populateMorphoDef(doc, morphoDef);
    xmlFreeDoc(doc);
  }
}

/**
 *
 */
void XmlMorphoDefFormatter::populateMorphoDef(xmlDocPtr doc,
					      MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(doc);

  xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "/MorphoDef",
                                         xpContext);
  if (xpObj != NULL) {
    xmlNodeSetPtr nodeSet = xpObj->nodesetval;
    if (nodeSet != NULL) {
      {for (int i = 0; i < nodeSet->nodeNr; i++) {
        if (langOk(nodeSet->nodeTab[i], morphoDef)) {
          addMorphoDef(nodeSet->nodeTab[i], morphoDef);
        }
      }}
    }
    xmlXPathFreeObject(xpObj);
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void XmlMorphoDefFormatter::addMorphoDef(xmlNodePtr node,
					 MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string defaultPluginId;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "pluginId");
    if (tmp != NULL) {
      defaultPluginId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  // Get classes
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Class", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          if (langOk(nodeSet->nodeTab[i], morphoDef)) {
            addClass(nodeSet->nodeTab[i],
		     defaultPluginId,
		     morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void XmlMorphoDefFormatter::addClass(xmlNodePtr node,
				     const string &parentDefaultPluginId,
				     MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string classId;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "id");
    if (tmp != NULL) {
      classId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string defaultPluginId = parentDefaultPluginId;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "pluginId");
    if (tmp != NULL) {
      defaultPluginId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  MorphoDef::Class &classDef = morphoDef.createClass(classId);

  // Get cases
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Case",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          if (langOk(nodeSet->nodeTab[i], morphoDef)) {
            addCase(nodeSet->nodeTab[i],
		    classDef,
		    morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  // Get declinations
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Declination",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          if (langOk(nodeSet->nodeTab[i], morphoDef)) {
            addDeclination(nodeSet->nodeTab[i],
			   classDef,
			   defaultPluginId,
			   morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void
XmlMorphoDefFormatter::
addCase(xmlNodePtr node,
	MorphoDef::Class &classDef,
	MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string id;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "id");
    if (tmp != NULL) {
      id = (char *) tmp;
      xmlFree(tmp);
    }
  }

  MorphoDef::Case &caseDef = classDef.createCase(id);

  // Get features
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Feature",
					   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  if (langOk(nodeSet->nodeTab[i], morphoDef)) {
	    xmlChar *tmp = xmlGetProp(nodeSet->nodeTab[i], (xmlChar *) "name");
	    if (tmp != NULL) {
	      caseDef.createFeature((char *) tmp);
	      xmlFree(tmp);
	    }
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  // Get cascades
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Cascade",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          if (langOk(nodeSet->nodeTab[i], morphoDef)) {
            addCascade(nodeSet->nodeTab[i], caseDef, morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void
XmlMorphoDefFormatter::
addDeclination(xmlNodePtr node,
               MorphoDef::Class &classDef,
	       const string &parentDefaultPluginId,
               MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string declinationCode;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "code");
    if (tmp != NULL) {
      declinationCode = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string defaultPluginId = parentDefaultPluginId;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "pluginId");
    if (tmp != NULL) {
      defaultPluginId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  MorphoDef::Declination &declinationDef =
    classDef.createDeclination(declinationCode);

  // Get stem trigger
  {
    bool hasStemTrigger = false;

    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "StemTrigger",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL && nodeSet->nodeNr > 0) {
        xmlNodePtr triggerNode = nodeSet->nodeTab[0];
        if (langOk(triggerNode, morphoDef)) {

	  string pluginId = defaultPluginId;
          {
            xmlChar *tmp = xmlGetProp(triggerNode, (xmlChar *) "pluginId");
            if (tmp != NULL) {
	      pluginId = (char *) tmp;
              xmlFree(tmp);
            }
          }

          {
            xmlChar *tmp = xmlNodeGetContent(triggerNode);
            if (tmp != NULL) {
	      declinationDef.setStemTrigger(pluginId, (char *) tmp);
	      hasStemTrigger = true;
              xmlFree(tmp);
            }
          }
          
        }
      }
      xmlXPathFreeObject(xpObj);
    }

    if (!hasStemTrigger) {
      declinationDef.setStemTrigger(defaultPluginId, "");
    }
  }

  // Get inflections
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Inflection",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr inflectionNode = nodeSet->nodeTab[i];
          if (langOk(inflectionNode, morphoDef)) {

            const MorphoDef::Case *caseDef = NULL;
            {
              xmlChar *tmp = xmlGetProp(inflectionNode,
                                        (xmlChar *) "caseId");
              if (tmp != NULL) {
                caseDef = classDef.getCase((char *) tmp);
                xmlFree(tmp);
              }
            }

	    if (caseDef == NULL) {
	      //ERROR
	    } else {
	      addInflection(inflectionNode,
			    declinationDef.createInflection(*caseDef),
			    defaultPluginId,
			    morphoDef);
	    }
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  // Get derivations
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Derivation",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          xmlNodePtr derivationNode = nodeSet->nodeTab[i];

          if (langOk(derivationNode, morphoDef)) {

            string classId;
            {
              xmlChar *tmp = xmlGetProp(derivationNode,
                                        (xmlChar *) "classId");
              if (tmp != NULL) {
                classId = (char *) tmp;
                xmlFree(tmp);
              }
            }

            string declinationCode;
            {
              xmlChar *tmp = xmlGetProp(derivationNode,
                                        (xmlChar *) "declinationCode");
              if (tmp != NULL) {
                declinationCode = (char *) tmp;
                xmlFree(tmp);
              }
            }

            string posName;
            {
              xmlChar *tmp = xmlGetProp(derivationNode,
                                        (xmlChar *) "pos");
              if (tmp != NULL) {
                posName = (char *) tmp;
                xmlFree(tmp);
              }
            }

            addInflection(derivationNode,
                          declinationDef.createDerivation(posName,
							  classId,
                                                          declinationCode),
			  defaultPluginId,
                          morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void
XmlMorphoDefFormatter::
addInflection(xmlNodePtr node,
              MorphoDef::Inflection &inflectionDef,
	      const string &defaultPluginId,
              MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  // Get form information
  {
    bool hasFormGenerator = false;

    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Form",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL && nodeSet->nodeNr > 0) {
        xmlNodePtr formNode = nodeSet->nodeTab[0];
        if (langOk(formNode, morphoDef)) {

	  string pluginId = defaultPluginId;
          {
            xmlChar *tmp = xmlGetProp(formNode, (xmlChar *) "pluginId");
            if (tmp != NULL) {
	      pluginId = (char *) tmp;
              xmlFree(tmp);
            }
          }

          {
            xmlChar *tmp = xmlNodeGetContent(formNode);
            if (tmp != NULL) {
	      inflectionDef.setFormGenerator(pluginId, (char *) tmp);
	      hasFormGenerator = true;
              xmlFree(tmp);
            }
          }

          
        }
      }
      xmlXPathFreeObject(xpObj);
    }

    if (!hasFormGenerator) {
      inflectionDef.setFormGenerator(defaultPluginId, "");
    }
  }

  // Get direct features
//   {
//     xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Feature",
//                                            xpContext);
//     if (xpObj != NULL) {
//       xmlNodeSetPtr nodeSet = xpObj->nodesetval;
//       if (nodeSet != NULL) {
//         {for (int i = 0; i < nodeSet->nodeNr; i++) {
//           if (langOk(nodeSet->nodeTab[i], morphoDef)) {
//             addFeature(nodeSet->nodeTab[i], inflectionDef, morphoDef);
//           }
//         }}
//       }
//       xmlXPathFreeObject(xpObj);
//     }
//   }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void
XmlMorphoDefFormatter::
addFeature(xmlNodePtr node,
           MorphoDef::Case &caseDef,
           MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string featureName;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "name");
    if (tmp != NULL) {
      featureName = (char *) tmp;
      xmlFree(tmp);
    }
  }

  MorphoDef::Feature &featureDef =
    caseDef.createFeature(featureName);

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void XmlMorphoDefFormatter::addCascade(xmlNodePtr node,
				       MorphoDef::Case &caseDef,
				       MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string classId;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "classId");
    if (tmp != NULL) {
      classId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string declinationCode;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "declinationCode");
    if (tmp != NULL) {
      declinationCode = (char *) tmp;
      xmlFree(tmp);
    }
  }

  MorphoDef::Cascade &cascadeDef =
    caseDef.createCascade(classId, declinationCode);

  // Get cascades
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Cascade",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          if (langOk(nodeSet->nodeTab[i], morphoDef)) {
            addCascade(nodeSet->nodeTab[i], cascadeDef, morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
void
XmlMorphoDefFormatter::addCascade(xmlNodePtr node,
				  MorphoDef::Cascade &cascadeDef,
				  MorphoDef &morphoDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(node->doc);
  xpContext->node = node;

  string classId;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "classId");
    if (tmp != NULL) {
      classId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string declinationCode;
  {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "declinationCode");
    if (tmp != NULL) {
      declinationCode = (char *) tmp;
      xmlFree(tmp);
    }
  }

  MorphoDef::Cascade &newCascadeDef =
    cascadeDef.createCascade(classId, declinationCode);

  // Get cascades
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Cascade",
                                           xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
        {for (int i = 0; i < nodeSet->nodeNr; i++) {
          if (langOk(nodeSet->nodeTab[i], morphoDef)) {
            addCascade(nodeSet->nodeTab[i], newCascadeDef, morphoDef);
          }
        }}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);
}

/**
 *
 */
bool XmlMorphoDefFormatter::langOk(xmlNodePtr node,
				   const MorphoDef &morphoDef) const {
  bool ok = true;
  if (node != NULL) {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "lang");
    if (tmp != NULL) {
      string nodeLang = (char *) tmp;
      xmlFree(tmp);

      ok = langOk(nodeLang, morphoDef.getIsoLanguageCode());
    }
  }
  return ok;
}

/**
 *
 */
bool XmlMorphoDefFormatter::langOk(const string &nodeLang,
				   const string &lang) const {
  bool ok = true;
  if (lang != "" && nodeLang != "") {
    string::size_type index = nodeLang.find(lang);
    if (index == string::npos) {
      // Not in list
      ok = false;

      // See if negation
      if (nodeLang.find('!') != string::npos) {
        ok = true;
      }
          
    } else {
      // In list
      if (index > 0 && nodeLang[index - 1] == '!') {
        // Not this language
        ok = false;
      }
    }
  }
  return ok;
}
