#ifndef _MORPHOLOGYDEFINITION_MORPHODEF_H_
#define _MORPHOLOGYDEFINITION_MORPHODEF_H_

/** \mainpage
 * Module that supplies a data structure for inflection.
 *
 * The morphology is described by the class MorphoDef.
 *
 * The actual inflection is performed by the class Inflector.
 */

#include <string>
#include <vector>
#include <set>
#include <map>

#include "MorphologyDefinition/FormGenerator.h"

namespace MorphologyDefinition {

  /**
   * Describes the morphology for a language.
   *
   * The inflection is handled by plugins. The default plugin assumes
   * simple suffixation.
   */
  class MorphoDef {
  public:

    /**
     *
     */
    MorphoDef(const std::string &isoLanguageCode = "");

    /**
     *
     */
    ~MorphoDef();

    /**
     *
     */
    MorphoDef(const MorphoDef &);

    /**
     *
     */
    const std::string &getIsoLanguageCode() const;

    /**
     *
     */
    class Feature {
    public:

      /**
       *
       */
      Feature(const std::string &name);

      /**
       *
       */
      ~Feature();

      /**
       *
       */
      const std::string &getName() const;

    private:
      std::string d_name;
    };

    /**
     *
     */
    class Cascade {
    public:

      /**
       *
       */
      Cascade(const std::string &classId,
	      const std::string &declinationCode);

      /**
       *
       */
      ~Cascade();

      /**
       *
       */
      Cascade(const Cascade &);

      /**
       *
       */
      const std::string &getClassId() const;

      /**
       *
       */
      const std::string &getDeclinationCode() const;

      /**
       *
       */
      Cascade &createCascade(const std::string &classId,
			     const std::string &declinationCode);

      /**
       *
       */
      typedef std::vector<const Cascade *> CascadeList;
      typedef CascadeList::const_iterator CascadeIterator;

      /**
       *
       */
      CascadeIterator cascadesBegin() const;

      /**
       *
       */
      CascadeIterator cascadesEnd() const;

    private:
      std::string d_classId;
      std::string d_declinationCode;
      CascadeList d_cascades;
    };

    class Class;
    class Inflection;

    /**
     *
     */
    class Case {
      friend class Class;
    public: 

      /**
       *
       */
      Case(const Class &, const std::string &id);

      /**
       *
       */
      ~Case();

      /**
       *
       */
      Case(const Case &);

      /**
       *
       */
      Case(const Class &, const Case &);

      /**
       *
       */
      const Class &getClass() const;

      /**
       *
       */
      const std::string &getId() const;

      /**
       *
       */
      Feature &createFeature(const std::string &name);

      /**
       *
       */
      bool hasFeature(const std::string &name) const;

      /**
       *
       */
      typedef std::vector<const Feature *> FeatureList;
      typedef FeatureList::const_iterator FeatureIterator;

      /**
       *
       */
      FeatureIterator featuresBegin() const;

      /**
       *
       */
      FeatureIterator featuresEnd() const;

      /**
       *
       */
      int getNbrOfFeatures() const;

      /**
       *
       */
      typedef std::vector<const Cascade *> CascadeList;
      typedef CascadeList::const_iterator CascadeIterator;

      /**
       *
       */
      Cascade &createCascade(const std::string &classId,
			     const std::string &declinationCode);

      /**
       *
       */
      CascadeIterator cascadesBegin() const;

      /**
       *
       */
      CascadeIterator cascadesEnd() const;

      /**
       *
       */
      typedef std::multimap<const Case *,
			    const Inflection *>::const_iterator
      InflectionIterator;

      /**
       *
       */
      InflectionIterator inflectionsBegin() const;

      /**
       *
       */
      InflectionIterator inflectionsEnd() const;

    private:

      /**
       *
       */
      void addFeature(Feature *);

    private:
      const Class &d_parent;
      std::string d_id;
      FeatureList d_features;
      CascadeList d_cascades;

      typedef std::map<std::string, const Feature *> FeatureNameMap;
      FeatureNameMap d_featureNameMap;
    };

    class Declination;

    /**
     *
     */
    class Inflection {
    public:

      /**
       *
       */
      Inflection(const Declination &, const Case *);

      /**
       *
       */
      virtual ~Inflection();

      /**
       *
       */
      Inflection(const Inflection &);

      /**
       *
       */
      const Declination &getDeclination() const;

      /**
       *
       */
      const Case *getCase() const;

      /**
       *
       */
      void setFormGenerator(const std::string &pluginId,
			    const std::string &value);

      /**
       *
       */
      const AbstractFormGenerator *getFormGenerator() const;

    private:
      const Declination &d_parent;
      const Case *d_case;

      AbstractFormGenerator *d_formGenerator;
    };

    /**
     *
     */
    class Derivation : public Inflection {
    public:

      /**
       *
       */
      Derivation(const class Declination &,
		 const std::string &posName,
		 const std::string &classId,
		 const std::string &declinationCode);

      /**
       *
       */
      ~Derivation();

      /**
       *
       */
      Derivation(const Derivation &);

      /**
       *
       */
      const std::string &getPosName() const;

      /**
       *
       */
      const std::string &getClassId() const;

      /**
       *
       */
      const std::string &getDeclinationCode() const;

      /**
       *
       */
      Feature &createFeature(const std::string &name);

      /**
       *
       */
      typedef std::vector<const Feature *> FeatureList;
      typedef FeatureList::const_iterator FeatureIterator;

      /**
       *
       */
      FeatureIterator featuresBegin() const;

      /**
       *
       */
      FeatureIterator featuresEnd() const;

    private:
      std::string d_posName;
      std::string d_classId;
      std::string d_declinationCode;
      FeatureList d_features;
    };

    /**
     *
     */
    class Declination {
    public:

      /**
       *
       */
      Declination(MorphoDef &, const std::string &code);

      /**
       *
       */
      ~Declination();

      /**
       *
       */
      Declination(const Declination &);

      /**
       *
       */
      const std::string &getCode() const;

      /**
       *
       */
      const MorphoDef &getMorphoDef() const;

      /**
       *
       */
      Inflection &createInflection(const Case &);

      /**
       *
       */
      typedef std::vector<const Inflection *> InflectionList;
      typedef InflectionList::const_iterator InflectionIterator;

      /**
       *
       */
      InflectionIterator inflectionsBegin() const;

      /**
       *
       */
      InflectionIterator inflectionsEnd() const;

      /**
       *
       */
      Derivation &createDerivation(const std::string &posName,
				   const std::string &classId,
				   const std::string &declinationCode);

      /**
       *
       */
      typedef std::vector<const Derivation *> DerivationList;
      typedef DerivationList::const_iterator DerivationIterator;

      /**
       *
       */
      DerivationIterator derivationsBegin() const;

      /**
       *
       */
      DerivationIterator derivationsEnd() const;

      /**
       *
       */
      void setStemTrigger(const std::string &pluginId,
			  const std::string &value);

      /**
       *
       */
      const AbstractStemTrigger *getStemTrigger() const;

    private:
      MorphoDef &d_parent;

      std::string d_code;

      AbstractStemTrigger *d_stemTrigger;

      InflectionList d_inflections;
      DerivationList d_derivations;
    };

    /**
     * A class is almost a pos, but not quite.
     */
    class Class {
      friend class Case;
    public:
   
      /**
       *
       */
      Class(MorphoDef &, const std::string &id);

      /**
       *
       */
      ~Class();

      /**
       *
       */
      Class(const Class &);

      /**
       *
       */
      const std::string &getId() const;

      /**
       *
       */
      Declination &createDeclination(const std::string &code);

      /**
       *
       */
      const Declination *getDeclination(const std::string &code) const;

      /**
       *
       */
      const std::vector<const Declination *> *
      getDeclinations(const std::string &code) const;

      /**
       *
       */
      typedef std::vector<const Declination *> DeclinationList;
      typedef DeclinationList::const_iterator DeclinationIterator;

      /**
       *
       */
      DeclinationIterator declinationsBegin() const;

      /**
       *
       */
      DeclinationIterator declinationsEnd() const;

      /**
       *
       */
      Case &createCase(const std::string &id);

      /**
       *
       */
      const Case *getCase(const std::string &id) const;

      /**
       *
       */
      typedef std::vector<const Case *> CaseList;
      typedef CaseList::const_iterator CaseIterator;

      /**
       *
       */
      CaseIterator casesBegin() const;

      /**
       *
       */
      CaseIterator casesEnd() const;

    private:

      /**
       *
       */
      void addDeclination(Declination *);

      /**
       *
       */
      void addCase(Case *);

    private:
      MorphoDef &d_parent;

      std::string d_id;
      DeclinationList d_declinations;
      std::map<std::string,
	       std::vector<const Declination *> *> d_declinationNameMap;

      CaseList d_cases;
      std::map<std::string, Case *> d_caseIdMap;

      typedef std::multimap<const Case *,
			    const Inflection *> CaseToInflectionsMap;
      CaseToInflectionsMap d_caseToInflections;
    };
  
    /**
     *
     */
    Class &createClass(const std::string &id);

    /**
     *
     */
    const Class *getClass(const std::string &id) const;

    /**
     *
     */
    typedef std::vector<const Class *> ClassList;
    typedef ClassList::const_iterator ClassIterator;

    /**
     *
     */
    ClassIterator classesBegin() const;

    /**
     *
     */
    ClassIterator classesEnd() const;

    /**
     * This class assumes ownership
     */
    void registerFormGeneratorFactory(AbstractFormGeneratorFactory *,
				      const std::string &id);

    /**
     *
     */
    const AbstractFormGeneratorFactory *
    getFormGeneratorFactory(const std::string &id) const;

  private:
    std::string d_isoLanguageCode;

    ClassList d_classes;
    std::map<std::string, Class *> d_classIdMap;

    std::map<std::string, AbstractFormGeneratorFactory *>
    d_formGeneratorFactories;
  };

}

#endif //_MORPHOLOGYDEFINITION_MORPHODEF_H_
