#include "MorphologyDefinition/Inflector.h"

using namespace std;
using namespace MorphologyDefinition;

/**
 *
 */
Inflector::Inflector(const MorphoDef &morphoDef) :
  d_morphoDef(morphoDef) {
}

/**
 *
 */
Inflector::~Inflector() {
}

/**
 *
 */
const MorphoDef &Inflector::getMorphoDef() const {
  return d_morphoDef;
}

/**
 *
 */
Inflector::Inflection::Inflection(const string &form) :
  d_form(form) {
}

/**
 *
 */
Inflector::Inflection::
Inflection(const string &form,
           const MorphoDef::Case &caseDef) :
  d_form(form) {
  addFeatures(caseDef);
}

/**
 *
 */
Inflector::Inflection::Inflection(const Inflection &o) :
  d_form(o.d_form),
  d_features(o.d_features) {
}

/**
 *
 */
Inflector::Inflection::~Inflection() {
}

/**
 *
 */
Inflector::Inflections::
Inflections(const Inflector &inflector,
            const string &lemma,
            const MorphoDef::Declination &declinationDef) :
  d_inflector(inflector),
  d_lemma(lemma),
  d_declinationDef(declinationDef) {
}

/**
 *
 */
Inflector::Inflection::FeatureIterator
Inflector::Inflection::featuresBegin() const {
  return d_features.begin();
}

/**
 *
 */
Inflector::Inflection::FeatureIterator
Inflector::Inflection::featuresEnd() const {
  return d_features.end();
}

/**
 *
 */
Inflector::Inflections::~Inflections() {
  {for (InflectionIterator it = inflectionsBegin();
        it != inflectionsEnd(); ++it) {
    delete *it;
  }}

  {for (DerivationIterator it = derivationsBegin();
        it != derivationsEnd(); ++it) {
    delete *it;
  }}
}

/**
 *
 */
Inflector::Inflections::Inflections(const Inflections &o) :
  d_inflector(o.d_inflector),
  d_lemma(o.d_lemma),
  d_declinationDef(o.d_declinationDef) {

  {for (InflectionIterator it = o.inflectionsBegin();
        it != o.inflectionsEnd(); ++it) {
    d_inflections.push_back(new Inflection(**it));
  }}

  {for (DerivationIterator it = o.derivationsBegin();
        it != o.derivationsEnd(); ++it) {
    d_derivations.push_back(new Derivation(**it));
  }}
}

/**
 *
 */
const Inflector &Inflector::Inflections::getInflector() const {
  return d_inflector;
}

/**
 *
 */
const string &Inflector::Inflection::getForm() const {
  return d_form;
}

/**
 *
 */
const string &Inflector::Inflections::getLemma() const {
  return d_lemma;
}

/**
 *
 */
void
Inflector::Inflection::addFeature(const MorphoDef::Feature *f) {
  d_features.push_back(f);
}

/**
 *
 */
void
Inflector::Inflection::
addFeatures(const MorphoDef::Case &caseDef) {
  {for (MorphoDef::Case::FeatureIterator
	  it = caseDef.featuresBegin();
	it != caseDef.featuresEnd(); ++it) {
    d_features.push_back(*it);
  }}
}

/**
 *
 */
const MorphoDef::Declination &
Inflector::Inflections::getDeclinationDef() const {
  return d_declinationDef;
}

/**
 *
 */
Inflector::Inflection &
Inflector::Inflections::createInflection(const string &form) {
  Inflection *newInflection = new Inflection(form);
  d_inflections.push_back(newInflection);
  return *newInflection;
}

/**
 *
 */
Inflector::Inflection &
Inflector::Inflections::
createInflection(const string &form,
                 const MorphoDef::Inflection &inflectionDef) {
  Inflection *newInflection = new Inflection(form, *inflectionDef.getCase());
  d_inflections.push_back(newInflection);
  return *newInflection;
}

/**
 *
 */
Inflector::Inflections::InflectionIterator
Inflector::Inflections::inflectionsBegin() const {
  return d_inflections.begin();
}

/**
 *
 */
Inflector::Inflections::InflectionIterator
Inflector::Inflections::inflectionsEnd() const {
  return d_inflections.end();
}

/**
 *
 */
Inflector::Derivation &
Inflector::Inflections::
createDerivation(const string &lemma,
                 const MorphoDef::Derivation &derivationDef) {
  const string &posName = derivationDef.getPosName();
  const string &classId = derivationDef.getClassId();
  const string &declinationCode = derivationDef.getDeclinationCode();

  const MorphoDef::Class *classDef =
    d_inflector.getMorphoDef().getClass(classId);
  if (classDef == NULL) {
    throw string("Colud not find (der) class definition ") + classId;
  }

  const MorphoDef::Declination *declinationDef =
    classDef->getDeclination(declinationCode);
  if (declinationDef == NULL) {
    throw string("Could not find (der) declination definition ") + 
      declinationCode;
  }

  Derivation *newDerivation =
    new Derivation(d_inflector, posName, lemma, *declinationDef);

  {for (MorphoDef::Derivation::FeatureIterator
	  it = derivationDef.featuresBegin();
	it != derivationDef.featuresEnd(); ++it) {
    newDerivation->addFeature(*it);
  }}

  d_derivations.push_back(newDerivation);
  return *newDerivation;
}

/**
 *
 */
Inflector::Inflections::DerivationIterator
Inflector::Inflections::derivationsBegin() const {
  return d_derivations.begin();
}

/**
 *
 */
Inflector::Inflections::DerivationIterator
Inflector::Inflections::derivationsEnd() const {
  return d_derivations.end();
}

/**
 *
 */
Inflector::Inflections
Inflector::inflect(const string &lemma,
                   const MorphoDef::Declination &declinationDef)
  const {
  Inflections inflections(*this, lemma, declinationDef);
  inflect(inflections);
  return inflections;
}

/**
 *
 */
Inflector::Inflections
Inflector::inflect(const string &lemma,
                   const string &classId,
                   const string &declinationCode) const {
  const MorphoDef::Class *classDef = d_morphoDef.getClass(classId);
  if (classDef == NULL) {
    throw string("Could not find class definition ") + classId;
  }

  const vector<const MorphoDef::Declination *> *declinationDefs =
    classDef->getDeclinations(declinationCode);
  if (declinationDefs == NULL) {
    throw string("Could not find declination definition ") + declinationCode;
  }
  
  // There might be several Declination objects for each code
  {
    vector<const MorphoDef::Declination *>::const_iterator
      it = declinationDefs->begin();
    while (it != declinationDefs->end()) {

      Inflections inflections(*this, lemma, **it);
      inflect(inflections);

      ++it;

      if (inflections.inflectionsBegin() != inflections.inflectionsEnd() ||
	  it == declinationDefs->end()) {
	return inflections;
      }
    }
  }

  throw "";
}

/**
 *
 */
bool Inflector::inflect(Inflector::Inflections &result) const {
  const MorphoDef::Declination &declinationDef =
    result.getDeclinationDef();

  const AbstractStemTrigger *stemTrigger =
    declinationDef.getStemTrigger();

  if (stemTrigger == NULL) {
    throw "Could not retreive stem trigger";
  }

  AbstractStem *stem = stemTrigger->trigger(result.getLemma());

  if (stem != NULL) {
    auto_ptr<AbstractStem> stemPtr(stem);

    {for (MorphoDef::Declination::InflectionIterator
            it = declinationDef.inflectionsBegin();
          it != declinationDef.inflectionsEnd(); ++it) {
      const MorphoDef::Inflection *inflectionDef = *it;
      const MorphoDef::Case *caseDef = inflectionDef->getCase();

      const AbstractFormGenerator *formGenerator =
	inflectionDef->getFormGenerator();

      if (formGenerator == NULL) {
	throw "Could not retreive form generator";
      }

      string form = formGenerator->generateForm(*stem);

      
      if (caseDef->cascadesBegin() == caseDef->cascadesEnd()) {
	result.createInflection(form, *inflectionDef);
      } else {
	vector<const MorphoDef::Feature *> cascadeFeatures;
	{for (MorphoDef::Case::FeatureIterator
		it2 = caseDef->featuresBegin();
	      it2 != caseDef->featuresEnd(); ++it2) {
	  cascadeFeatures.push_back(*it2);
	}}

        {for (MorphoDef::Case::CascadeIterator
                it2 = caseDef->cascadesBegin();
              it2 != caseDef->cascadesEnd(); ++it2) {
          const MorphoDef::Cascade *cascadeDef = *it2;
          inflectCascade(form, result, *cascadeDef, cascadeFeatures);
        }}
      }
    }}

    {for (MorphoDef::Declination::DerivationIterator
            it = declinationDef.derivationsBegin();
          it != declinationDef.derivationsEnd(); ++it) {
      const MorphoDef::Derivation *derivationDef = *it;

      const AbstractFormGenerator *formGenerator =
	derivationDef->getFormGenerator();

      if (formGenerator == NULL) {
	throw "Could not retreive form generator";
      }

      string form = formGenerator->generateForm(*stem);

      Derivation &derivation =
	result.createDerivation(form, *derivationDef);

      inflect(derivation.getModifiableInflections());
    }}

    return true;
  }

  return false;
}

/**
 *
 */
void
Inflector::
inflectCascade(const string &form,
               Inflections &result,
               const MorphoDef::Cascade &cascadeDef,
	       vector<const MorphoDef::Feature *> features)
  const {

  {
    const string &classId = cascadeDef.getClassId();
    const string &declinationCode = cascadeDef.getDeclinationCode();

    const MorphoDef::Class *classDef =
      d_morphoDef.getClass(classId);
    if (classDef == NULL) {
      throw string("Could not find (cascade) class definition ") + classId;
    } else {
      const vector<const MorphoDef::Declination *> *declinationDefs =
	classDef->getDeclinations(declinationCode);
      if (declinationDefs == NULL) {
	throw string("Could not find (cascade) declination definition ") +
	  declinationCode;
      }
  
      // There might be several Declination objects for each code
      {
	vector<const MorphoDef::Declination *>::const_iterator
	  it = declinationDefs->begin();
	while (it != declinationDefs->end()) {
	  const MorphoDef::Declination *declinationDef = *it;

	  Inflections inflections(*this, form, *declinationDef);
	  inflect(inflections);

	  ++it;
	
	  if (inflections.inflectionsBegin() != inflections.inflectionsEnd() ||
	      it == declinationDefs->end()) {
	    {for (Inflector::Inflections::InflectionIterator
		    it = inflections.inflectionsBegin();
		  it != inflections.inflectionsEnd(); ++it) {
	      const Inflector::Inflection *inflection = *it;

	      {for (Inflector::Inflection::FeatureIterator
		    it = inflection->featuresBegin();
		    it != inflection->featuresEnd(); ++it) {
		features.push_back(*it);
	      }}

	      if (cascadeDef.cascadesBegin() == cascadeDef.cascadesEnd()) {
		// Reached leaf
		Inflector::Inflection &cascadeInflection =
		  result.createInflection(inflection->getForm());
		{for (vector<const MorphoDef::Feature *>::iterator
			it = features.begin();
		      it != features.end(); ++it) {
		  cascadeInflection.addFeature(*it);
		}}
	      } else {
		{for (MorphoDef::Cascade::CascadeIterator
			it = cascadeDef.cascadesBegin();
		      it != cascadeDef.cascadesEnd(); ++it) {
		  inflectCascade(inflection->getForm(), result,
				 **it, features);
		}}
	      }
	    }}
	    break;
	  }
	}
      }
    }
  }
}

/**
 *
 */
Inflector::Derivation::Derivation(const Inflector &inflector,
				  const string &posName,
				  const string &lemma,
				  const MorphoDef::Declination &declDef) :
  Inflection(lemma),
  d_posName(posName),
  d_inflections(inflector, lemma, declDef) {
}

/**
 *
 */
Inflector::Derivation::~Derivation() {
}

/**
 *
 */
Inflector::Derivation::Derivation(const Derivation &o) :
  Inflection(o),
  d_posName(o.d_posName),
  d_inflections(o.d_inflections) {
}

/**
 *
 */
const string &Inflector::Derivation::getPosName() const {
  return d_posName;
}

/**
 *
 */
Inflector::Inflections &Inflector::Derivation::getModifiableInflections() {
  return d_inflections;
}

/**
 *
 */
const Inflector::Inflections &Inflector::Derivation::getInflections() const {
  return d_inflections;
}
