#include "LinguisticDefinition/XmlLingFeaturesFormatter.h"

using namespace std;
using namespace LinguisticDefinition;

/**
 *
 */
XmlLingFeaturesFormatter::XmlLingFeaturesFormatter(const LingDef &lingDef) :
  LingFeaturesFormatter(lingDef) {
}

/**
 *
 */
LingFeatures
XmlLingFeaturesFormatter::createLingFeatures(const string &xmlData) {
  LingFeatures features(getLingDef());
  populateLingFeatures(xmlData, features);
  return features;
}

/**
 *
 */
LingFeatures
XmlLingFeaturesFormatter::createLingFeatures(xmlNodePtr featuresNode) {
  LingFeatures features(getLingDef());
  populateLingFeatures(featuresNode, features);
  return features;
}

/**
 *
 */
bool XmlLingFeaturesFormatter::populateLingFeatures(const string &xmlData,
						    LingFeatures &features) {
  bool populationOk = false;
  xmlDocPtr doc = xmlParseDoc((xmlChar *) xmlData.c_str());
  if (doc != NULL) {
    xmlXPathContextPtr xpContext = xmlXPathNewContext(doc);
    xmlXPathObjectPtr xpObj =
      xmlXPathEvalExpression((xmlChar *) "//LingFeatures", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL && nodeSet->nodeNr > 0) {
	populateLingFeatures(nodeSet->nodeTab[0], features);
      }
      xmlXPathFreeObject(xpObj);
    }
    
    xmlXPathFreeContext(xpContext);
    xmlFreeDoc(doc);
  }
  return populationOk;
}

/**
 *
 */
bool XmlLingFeaturesFormatter::populateLingFeatures(xmlNodePtr featuresNode,
						    LingFeatures &features) {
  bool populationOk = true;

  const LingDef::Pos *posDef = NULL;
  {
    xmlChar *tmp = xmlGetProp(featuresNode, (xmlChar *) "pos");
    if (tmp != NULL) {
      posDef = getLingDef().getPos((char *) tmp);
      xmlFree(tmp);
    }
  }

  if (posDef == NULL) {
    // ERROR
    populationOk = false;

  } else {
    features.setPosDef(*posDef);

    xmlXPathContextPtr xpContext = xmlXPathNewContext(featuresNode->doc);
    xpContext->node = featuresNode;

    features.setDefaults();

    {
      xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Lemma",
					     xpContext);
      if (xpObj != NULL) {
	xmlNodeSetPtr nodeSet = xpObj->nodesetval;
	if (nodeSet != NULL && nodeSet->nodeNr > 0) {
	  xmlChar *tmp = xmlNodeGetContent(nodeSet->nodeTab[0]);
	  if (tmp != NULL) {
	    features.setLemma((char *) tmp);
	    xmlFree(tmp);
	  }
	}
	xmlXPathFreeObject(xpObj);
      }
    }

    {
      xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Form",
					     xpContext);
      if (xpObj != NULL) {
	xmlNodeSetPtr nodeSet = xpObj->nodesetval;
	if (nodeSet != NULL && nodeSet->nodeNr > 0) {
	  xmlChar *tmp = xmlNodeGetContent(nodeSet->nodeTab[0]);
	  if (tmp != NULL) {
	    features.setForm((char *) tmp);
	    xmlFree(tmp);
	  }
	}
	xmlXPathFreeObject(xpObj);
      }
    }

    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Feature",
					   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr featureNode = nodeSet->nodeTab[i];
	  string featureName;
	  {
	    xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "name");
	    if (tmp != NULL) {
	      featureName = (char *) tmp;
	      xmlFree(tmp);
	    }
	  }

	  const LingDef::Feature *feature =
	    posDef->getFeature(featureName);
	  if (feature == NULL) {
	    // ERROR
	    populationOk = false;

	  } else {

	    if (feature->getType() == LingDef::Feature::BOOLEAN) {
	      features.set(*feature);

	    } else if (feature->getType() == LingDef::Feature::ENUM ||
		       feature->getType() ==
		       LingDef::Feature::REFERENCE) {
	      xmlXPathContextPtr xpContext2 =
		xmlXPathNewContext(featureNode->doc);
	      xpContext2->node = featureNode;
		    
	      xmlXPathObjectPtr xpObj2 =
		xmlXPathEval((xmlChar *) "Value", xpContext2);
	      if (xpObj2 != NULL) {
		xmlNodeSetPtr nodeSet2 = xpObj2->nodesetval;
		if (nodeSet2 != NULL) {
		  {for (int j = 0; j < nodeSet2->nodeNr; j++) {
		    xmlNodePtr valueNode = nodeSet2->nodeTab[j];
		    string valueValue;
		    {
		      xmlChar *tmp = xmlNodeGetContent(valueNode);
		      if (tmp != NULL) {
			valueValue = (char *) tmp;
			xmlFree(tmp);
		      }
		    }

		    if (feature->getType() == LingDef::Feature::ENUM) {
		      const LingDef::Feature *valueFeature =
			posDef->getFeature(valueValue);
		      if (valueFeature == NULL) {
			// ERROR
			populationOk = false;

		      } else {
			if (valueFeature->getParentEnum() != feature) {
			  // ERROR
			  populationOk = false;

			} else {
			  features.set(*valueFeature);
			}
		      }

		    } else {
		      int referenceValue = 0;
		      {
			string referenceValueS;
			if (valueValue != "" && valueValue[0] == '$') {
			  referenceValueS = valueValue.substr(1);
			} else {
			  referenceValueS = valueValue;
			}
			referenceValue = atoi(referenceValueS.c_str());
		      }

		      if (referenceValue != 0) {
			features.setReference(*feature, referenceValue);
		      }
		    }
		  }}
		}
		xmlXPathFreeObject(xpObj2);
	      }
	      xmlXPathFreeContext(xpContext2);
	    }
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
    xmlXPathFreeContext(xpContext);
  }

  return populationOk;
}

/**
 *
 */
void XmlLingFeaturesFormatter::output(const LingFeatures &features,
				      ostream &out) const {
  xmlOutputBufferPtr b = xmlAllocOutputBuffer(NULL);
  xmlTextWriterPtr writer = xmlNewTextWriter(b);
  output(features, writer);
  out << xmlBufferContent(b->buffer);
  xmlFreeTextWriter(writer);
}

/**
 *
 */
void XmlLingFeaturesFormatter::output(const LingFeatures &features,
				      xmlTextWriterPtr writer) const {
  xmlTextWriterStartElement(writer, (xmlChar *) "LingFeatures");

  xmlTextWriterWriteAttribute(writer,
			      (xmlChar *) "pos",
			      (xmlChar *) features.getPosDef()->
			      getName().c_str()); //TODO: validate ok

  if (features.getForm() != "") {
    xmlTextWriterStartElement(writer, (xmlChar *) "Form");
    xmlTextWriterWriteString(writer,
			     (xmlChar *) features.getForm().c_str());
    xmlTextWriterEndElement(writer);
  }

  if (features.getLemma() != "") {
    xmlTextWriterStartElement(writer, (xmlChar *) "Lemma");
    xmlTextWriterWriteString(writer,
			     (xmlChar *) features.getLemma().c_str());
    xmlTextWriterEndElement(writer);
  }

  {for (LingFeatures::FeatureIterator it = features.featuresBegin();
	it != features.featuresEnd(); ++it) {
    const LingDef::Feature *f = *it;

    if (f->getType() == LingDef::Feature::BOOLEAN) {
      const LingDef::Feature *enumFeature = f->getParentEnum();
      if (enumFeature == NULL) {
	xmlTextWriterStartElement(writer, (xmlChar *) "Feature");
	xmlTextWriterWriteAttribute(writer,
				    (xmlChar *) "name",
				    (xmlChar *) f->getName().c_str());
	xmlTextWriterEndElement(writer);
      }
    } else if (f->getType() == LingDef::Feature::ENUM) {
      xmlTextWriterStartElement(writer, (xmlChar *) "Feature");
      xmlTextWriterWriteAttribute(writer,
				  (xmlChar *) "name",
				  (xmlChar *) f->getName().c_str());

      set<const LingDef::Feature *> enumValues;
      features.getEnumValues(*f, enumValues);

      {for (set<const LingDef::Feature *>::iterator
	      it2 = enumValues.begin();
	    it2 != enumValues.end(); ++it2) {
	xmlTextWriterStartElement(writer, (xmlChar *) "Value");
	xmlTextWriterWriteString(writer,
				 (xmlChar *) (*it2)->getName().c_str());
	xmlTextWriterEndElement(writer);
      }}

      xmlTextWriterEndElement(writer);

    } else if (f->getType() == LingDef::Feature::REFERENCE) {
      xmlTextWriterStartElement(writer, (xmlChar *) "Feature");
      xmlTextWriterWriteAttribute(writer,
				  (xmlChar *) "name",
				  (xmlChar *) f->getName().c_str());

      vector<int> references;
      features.getReferenceList(*f, references);
      {for (vector<int>::iterator it2 = references.begin();
	    it2 != references.end(); ++it2) {
	stringstream s;
	// 	s << '$' << *it2;
	xmlTextWriterStartElement(writer, (xmlChar *) "Value");
	xmlTextWriterWriteString(writer, (xmlChar *) s.str().c_str());
	xmlTextWriterEndElement(writer);
      }}

      xmlTextWriterEndElement(writer);
    }
  }}

  xmlTextWriterEndElement(writer);
}

/**
 *
 */
string
XmlLingFeaturesFormatter::output(const LingFeatures &features) const {
  stringstream s;
  output(features, s);
  return s.str();
}
