#ifndef _LINGUISTICDEFINITION_XMLLINGDEFFORMATTER_H_
#define _LINGUISTICDEFINITION_XMLLINGDEFFORMATTER_H_

#include <map>

#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xmlwriter.h>

#include "LingDef.h"
#include "LingDefFormatter.h"

namespace LinguisticDefinition {

  /**
   * Interprets an XML document, and creates a LingDef object accordingly.
   *
   * An example XML:
   *
   * <pre><code>
   *  &lt;LingDef xml:lang="ar"&gt;
   *
   *   &lt;Pos id="base" type="virtual"&gt;
   *    &lt;Misc&gt;
   *     &lt;Enum name="langlevel"&gt;
   *      &lt;Value name="z0"/&gt;
   *      &lt;Value name="z1"/&gt;
   *      &lt;Value name="z2"/&gt;
   *     &lt;/Enum&gt;
   *     &lt;Boolean name="void_translation"/&gt;
   *    &lt;/Misc&gt;
   *   &lt;/Pos&gt;
   *
   *   &lt;Pos id="noun" inherits="base" type="element"&gt;
   *    &lt;Name short="N"&gt;NOUN&lt;/Name&gt;
   *     &lt;Morphology&gt;
   *      &lt;Enum name="gender" allowNone="false" allowSeveral="true"&gt;
   *       &lt;Value name="masc" short="m" default="true"/&gt;
   *       &lt;Value name="fem" short="f"/&gt;
   *      &lt;/Enum&gt;
   *   
   *      &lt;Enum name="number"&gt;
   *       &lt;Value name="sing" short="s" default="true"/&gt;
   *       &lt;Value name="plur" short="p"/&gt;
   *      &lt;/Enum&gt;
   *     &lt;/Morphology&gt;
   *   &lt;/Pos&gt;
   *
   *  &lt;/LingDef&gt;
   * </code></pre>
   *
   * For each node, it is possible to set the property <code>xml:lang</code>,
   * which allows deactivation of that node and its children if the language
   * does not match. The property can be either positive or negative: 'ar'
   * would cause the node to be used only for Arabic, '!ar' all languages
   * but Arabic, 'ar,fa,ur' Arabic, Farsi and Urdu, '!ar,!fa,!ur' all but
   * Arabic Farsi and Urdu.
   * 
   */
  class XmlLingDefFormatter : public LingDefFormatter {
  public:

    /**
     *
     */
    XmlLingDefFormatter();

    /**
     * Creates a LingDef object for an XML document
     * @param xmlData The XML document, as a string
     * @param isoLanguageCode The ISO language code for the language to use
     */
    LingDef createLingDef(const std::string &xmlData,
			  const std::string &isoLanguageCode = "");

    /**
     * Creates a LingDef object for an XML document
     * @param doc The XML document, as a constructed xmlDoc
     * @param isoLanguageCode The ISO language code for the language to use
     */
    LingDef createLingDef(xmlDocPtr doc,
			  const std::string &isoLanguageCode = "");

    /**
     * A quick access function, to avoid copying the result object.
     * @param xmlData The XML document, as a string
     * @param lingDef A pre-constructed LingDef object, which will be populated
     * @return Whether or not the document was correct
     */
    bool populateLingDef(const std::string &xmlData, LingDef &lingDef);

    /**
     * A quick access function, to avoid copying the result object.
     * @param doc The XML document, as a constructed xmlDoc
     * @param lingDef A pre-constructed LingDef object, which will be populated
     * @return Whether or not the document was correct
     */
    bool populateLingDef(xmlDocPtr doc, LingDef &lingDef);

    /**
     *
     */
    std::string output(const LingDef &) const;

    /**
     *
     */
    void output(const LingDef &, std::ostream &) const;

  private:

    /**
     *
     */
    bool langOk(const std::string &, const std::string &) const;

    /**
     *
     */
    bool langOk(xmlNodePtr, const LingDef &) const;

    /**
     *
     */
    bool addLingDef(xmlNodePtr, LingDef &);

    /**
     *
     */
    LingDef::Pos *addPos(xmlNodePtr, LingDef &, LingDef::Pos *parentPos);

    /**
     *
     */
    LingDef::Feature::Domain getFeatureDomain(xmlNodePtr featureNode);

    /**
     *
     */
    LingDef::Feature::Type getFeatureType(xmlNodePtr featureNode);

    /**
     *
     */
    bool addFeature(xmlNodePtr, LingDef::Pos &);

    /**
     *
     */
    bool addFeature(xmlNodePtr, LingDef::Feature &);

    /**
     *
     */
    bool addTree(xmlNodePtr, LingDef &);

    /**
     *
     */
    bool addTreeNode(xmlNodePtr, LingDef &, LingDef::Tree::Node &);

    /**
     *
     */
    void getPosIds(xmlNodePtr posNode,
		   std::string &id,
		   std::string &inheritsId);


  private:
  };

}

#endif //_LINGUISTICDEFINITION_XMLLINGDEFFORMATTER_H_
