#include "LinguisticDefinition/XmlLingDefFormatter.h"

using namespace std;
using namespace LinguisticDefinition;

/**
 *
 */
XmlLingDefFormatter::XmlLingDefFormatter() {
}

/**
 *
 */
LingDef XmlLingDefFormatter::createLingDef(const string &xmlData,
					   const string &isoLanguageCode) {
  LingDef lingDef(isoLanguageCode);
  xmlDocPtr doc = xmlParseDoc((xmlChar *) xmlData.c_str());
  if (doc != NULL) {
    populateLingDef(doc, lingDef);
    xmlFreeDoc(doc);
  }
  return lingDef;
}

/**
 *
 */
LingDef XmlLingDefFormatter::createLingDef(xmlDocPtr doc,
					   const string &isoLanguageCode) {
  LingDef lingDef(isoLanguageCode);
  populateLingDef(doc, lingDef);
  return lingDef;
}

/**
 *
 */
bool XmlLingDefFormatter::populateLingDef(const string &xmlData,
					  LingDef &lingDef) {
  bool populationOk = false;
  xmlDocPtr doc = xmlParseDoc((xmlChar *) xmlData.c_str());
  if (doc != NULL) {
    populationOk = populateLingDef(doc, lingDef);
    xmlFreeDoc(doc);
  }
  return populationOk;
}

/**
 *
 */
bool XmlLingDefFormatter::populateLingDef(xmlDocPtr doc,
					  LingDef &lingDef) {
  bool populationOk = true;

  xmlXPathContextPtr xpContext = xmlXPathNewContext(doc);

  xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "/LingDef", xpContext);
  if (xpObj != NULL) {
    xmlNodeSetPtr nodeSet = xpObj->nodesetval;
    if (nodeSet != NULL) {
      {for (int i = 0; i < nodeSet->nodeNr; i++) {
	if (langOk(nodeSet->nodeTab[i], lingDef)) {
	  addLingDef(nodeSet->nodeTab[i], lingDef);
	}
      }}
    }
    xmlXPathFreeObject(xpObj);
  }

  xmlXPathFreeContext(xpContext);

  return populationOk;
}

/**
 *
 */
bool XmlLingDefFormatter::langOk(xmlNodePtr node, const LingDef &lingDef) const {
  bool ok = true;
  if (node != NULL) {
    xmlChar *tmp = xmlGetProp(node, (xmlChar *) "lang");
    if (tmp != NULL) {
      string nodeLang = (char *) tmp;
      xmlFree(tmp);

      ok = langOk(nodeLang, lingDef.getIsoLanguageCode());
    }
  }
  return ok;
}

/**
 *
 */
bool XmlLingDefFormatter::langOk(const string &nodeLang,
				 const string &lang) const {
  bool ok = true;
  if (lang != "" && nodeLang != "") {
    string::size_type index = nodeLang.find(lang);
    if (index == string::npos) {
      // Not in list
      ok = false;

      // See if negation
      if (nodeLang.find('!') != string::npos) {
	ok = true;
      }
	  
    } else {
      // In list
      if (index > 0 && nodeLang[index - 1] == '!') {
	// Not this language
	ok = false;
      }
    }
  }
  return ok;
}

/**
 *
 */
bool XmlLingDefFormatter::addLingDef(xmlNodePtr lingDefNode, LingDef &lingDef) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(lingDefNode->doc);
  xpContext->node = lingDefNode;

  map<string, LingDef::Pos *> posIdMap;
  set<string> posIds;
  set<string> needsPosIds;

  // Get semantic trees
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Tree", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  if (langOk(nodeSet->nodeTab[i], lingDef)) {
	    addTree(nodeSet->nodeTab[i], lingDef);
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }
  
  // Get parts of speech
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Pos", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	bool done;
	do {
	  done = true;
	  {for (int i = 0; i < nodeSet->nodeNr; i++) {
	    xmlNodePtr posNode = nodeSet->nodeTab[i];
	    if (langOk(posNode, lingDef)) {
	      string posId;
	      string posInheritsId;
	      getPosIds(posNode, posId, posInheritsId);

	      posIds.insert(posId);
	      if (posInheritsId != "") {
		needsPosIds.insert(posInheritsId);
	      }

	      if (posIdMap[posId] == NULL) {
		bool parentOk = true;
		LingDef::Pos *parentPos = NULL;
		if (posInheritsId != "") {
		  parentPos = posIdMap[posInheritsId];
		  if (parentPos == NULL) {
		    parentOk = false;
		    done = false;
		  }
		}
		if (parentOk) {
		  posIdMap[posId] = addPos(posNode, lingDef, parentPos);
		}
	      }
	    }
	  }}

	  if (!done) {
	    // We are about to do another loop
	    // Check if the pos id's we need exist
	    {for (set<string>::iterator it = needsPosIds.begin();
		  it != needsPosIds.end(); ++it) {
	      if (posIds.find(*it) == posIds.end()) {
		cerr << "Super pos " << *it << " not defined" << endl;
		done = true;
	      }
	    }}
	  }

	} while (!done);
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);

  return true;
}

/**
 *
 */
void XmlLingDefFormatter::getPosIds(xmlNodePtr posNode,
				    string &id,
				    string &inheritsId) {
  string posId;
  {
    xmlChar *tmp = xmlGetProp(posNode, (xmlChar *) "id");
    if (tmp != NULL) {
      id = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string superPosId;
  {
    xmlChar *tmp = xmlGetProp(posNode, (xmlChar *) "inherits");
    if (tmp != NULL) {
      inheritsId = (char *) tmp;
      xmlFree(tmp);
    }
  }
}

/**
 *
 */
LingDef::Pos *XmlLingDefFormatter::addPos(xmlNodePtr posNode, LingDef &lingDef,
					  LingDef::Pos *parentPos) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(posNode->doc);
  xpContext->node = posNode;

  xmlNodePtr posNoteNode = NULL;
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Note", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  if (langOk(nodeSet->nodeTab[i], lingDef)) {
	    posNoteNode = nodeSet->nodeTab[i];
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlNodePtr posNameNode = NULL;
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Name", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  if (langOk(nodeSet->nodeTab[i], lingDef)) {
	    posNameNode = nodeSet->nodeTab[i];
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  string posId;
  {
    xmlChar *tmp = xmlGetProp(posNode, (xmlChar *) "id");
    if (tmp != NULL) {
      posId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string superPosId;
  {
    xmlChar *tmp = xmlGetProp(posNode, (xmlChar *) "inherits");
    if (tmp != NULL) {
      superPosId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  string posName;
  {
    if (posNameNode != NULL) {
      xmlChar *tmp = xmlNodeGetContent(posNameNode);
      if (tmp != NULL) {
	posName = (char *) tmp;
	xmlFree(tmp);
      }
    }
  }

  string posShortName;
  {
    if (posNameNode != NULL) {
      xmlChar *tmp = xmlGetProp(posNameNode, (xmlChar *) "short");
      if (tmp != NULL) {
	posShortName = (char *) tmp;
	xmlFree(tmp);
      }
    }
  }

  LingDef::Pos::Type posType = LingDef::Pos::ELEMENT;
  {
    xmlChar *tmp = xmlGetProp(posNode, (xmlChar *) "type");
    if (tmp != NULL) {
      if (!xmlStrcasecmp(tmp, (xmlChar *) "virtual")) {
	posType = LingDef::Pos::VIRTUAL;

      } else if (!xmlStrcasecmp(tmp, (xmlChar *) "syntagm")) {
	posType = LingDef::Pos::SYNTAGM;
      }
      xmlFree(tmp);
    }
  }

  // We now have enough information to create the Pos object
  LingDef::Pos *pos;
  if (posType == LingDef::Pos::VIRTUAL) {
    if (parentPos != NULL) {
      pos = &parentPos->createVirtualSubPos();
    } else {
      pos = &lingDef.createVirtualPos();
    }
  } else {
    if (parentPos != NULL) {
      pos = &parentPos->createSubPos(posName);
    } else {
      pos = &lingDef.createPos(posName);
    }
  }

  pos->setType(posType);

  if (posNoteNode != NULL) {
    {
      xmlChar *tmp = xmlNodeGetContent(posNoteNode);
      if (tmp != NULL) {
	pos->setNote((char *) tmp);
	xmlFree(tmp);
      }
    }
  }

  if (posShortName != "") {
    pos->setShortName(posShortName);
  }

  // Get features
  {
    xmlXPathObjectPtr xpObj =
      xmlXPathEval((xmlChar *) ".//Boolean|.//Enum|.//Reference|.//VTree",
		   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  if (langOk(nodeSet->nodeTab[i], lingDef)) {
	    addFeature(nodeSet->nodeTab[i], *pos);
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  // Get conflicts
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) ".//Conflict",
					   xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr conflictNode = nodeSet->nodeTab[i];
	  if (langOk(conflictNode, lingDef)) {

	    string aName;
	    {
	      xmlChar *tmp = xmlGetProp(conflictNode, (xmlChar *) "aName");
	      if (tmp != NULL) {
		aName = (char *) tmp;
		xmlFree(tmp);
	      }
	    }

	    string bName;
	    {
	      xmlChar *tmp = xmlGetProp(conflictNode, (xmlChar *) "bName");
	      if (tmp != NULL) {
		bName = (char *) tmp;
		xmlFree(tmp);
	      }
	    }

	    const LingDef::Feature *aFeature = pos->getFeature(aName);
	    const LingDef::Feature *bFeature = pos->getFeature(bName);

	    if (aFeature == NULL) {
	      //ERROR
	      cerr << "Feature '" << aName
		   << "' in conflict not defined " << endl;
	    } else if (bFeature == NULL) {
	      //ERROR
	      cerr << "Feature '" << bName
		   << "' in conflict not defined " << endl;
	    } else {
	      lingDef.addConflict(*aFeature, *bFeature);
	    }
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);

  return pos;
}

/**
 *
 */
LingDef::Feature::Domain
XmlLingDefFormatter::getFeatureDomain(xmlNodePtr featureNode) {
  string parentNodeName = (char *) featureNode->parent->name;
  if (parentNodeName == "Morphology") {
    return LingDef::Feature::MORPHO;
  } else if (parentNodeName == "Syntax") {
    return LingDef::Feature::SYNTAX;
  } else if (parentNodeName == "Semantics") {
    return LingDef::Feature::SEMANTIC;
  } else if (parentNodeName == "Misc") {
    return LingDef::Feature::MISC;
  }
  return LingDef::Feature::MISC;
}

/**
 *
 */
LingDef::Feature::Type
XmlLingDefFormatter::getFeatureType(xmlNodePtr featureNode) {
  string nodeName = (char *) featureNode->name;
  if (nodeName == "Boolean") {
    return LingDef::Feature::BOOLEAN;
  } else if (nodeName == "Enum") {
    return LingDef::Feature::ENUM;
  } else if (nodeName == "Reference") {
    return LingDef::Feature::REFERENCE;
  } else if (nodeName == "VTree") {
    return LingDef::Feature::VTREE;
  }
  return LingDef::Feature::BOOLEAN;
}

/**
 *
 */
bool XmlLingDefFormatter::addFeature(xmlNodePtr featureNode, LingDef::Pos &pos) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(featureNode->doc);
  xpContext->node = featureNode;

  string featureName;
  {
    xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "name");
    if (tmp != NULL) {
      featureName = (char *) tmp;
      xmlFree(tmp);
    }
  }

  addFeature(featureNode,
	     pos.createFeature(featureName,
			       getFeatureDomain(featureNode),
			       getFeatureType(featureNode)));

  xmlXPathFreeContext(xpContext);

  return true;
}

/**
 *
 */
bool XmlLingDefFormatter::addFeature(xmlNodePtr featureNode,
				     LingDef::Feature &feature) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(featureNode->doc);
  xpContext->node = featureNode;

  if (feature.getType() == LingDef::Feature::BOOLEAN) {

    {
      xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "default");
      if (tmp != NULL) {
	feature.setDefault(xmlStrcmp(tmp, (xmlChar *) "true") == 0);
	xmlFree(tmp);
      }
    }

    {
      xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "short");
      if (tmp != NULL) {
	feature.setShortName((char *) tmp);
	xmlFree(tmp);
      }
    }

  } else if (feature.getType() == LingDef::Feature::ENUM) {

    {
      xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "allowNone");
      if (tmp != NULL) {
	feature.setAllowNoValue(xmlStrcmp(tmp, (xmlChar *) "true") == 0);
	xmlFree(tmp);
      }
    }

    {
      xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "allowSeveral");
      if (tmp != NULL) {
	feature.setAllowSeveralValues(xmlStrcmp(tmp, (xmlChar *)"true") == 0);
	xmlFree(tmp);
      }
    }

    // Go through values
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Value", xpContext);
    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  xmlNodePtr valueNode = nodeSet->nodeTab[i];
	  if (langOk(valueNode, *feature.getLingDef())) {

	    string valueName;
	    {
	      xmlChar *tmp = xmlGetProp(valueNode, (xmlChar *) "name");
	      if (tmp != NULL) {
		valueName = (char *) tmp;
		xmlFree(tmp);
	      }
	    }

	    addFeature(valueNode, feature.createEnumValueFeature(valueName));
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }

  } else if (feature.getType() == LingDef::Feature::REFERENCE) {
    {
      xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "allowSeveral");
      if (tmp != NULL) {
	feature.setAllowSeveralValues(xmlStrcmp(tmp, (xmlChar *)"true") == 0);
	xmlFree(tmp);
      }
    }

  } else if (feature.getType() == LingDef::Feature::VTREE) {
    {
      xmlChar *tmp = xmlGetProp(featureNode, (xmlChar *) "treeId");
      if (tmp != NULL) {
	const LingDef::Tree *tree =
	  feature.getLingDef()->getTree((char *) tmp);
	if (tree == NULL) {
	  cerr << "Could not find tree " << (char *) tmp << endl;
	} else {
	  feature.setTree(*tree);
	}
	xmlFree(tmp);
      }
    }
  }

  xmlXPathFreeContext(xpContext);

  return true;
}

/**
 *
 */
bool XmlLingDefFormatter::addTree(xmlNodePtr treeNode, LingDef &lingDef) {
  string treeId;
  {
    xmlChar *tmp = xmlGetProp(treeNode, (xmlChar *) "id");
    if (tmp != NULL) {
      treeId = (char *) tmp;
      xmlFree(tmp);
    }
  }

  addTreeNode(treeNode, lingDef, lingDef.createTree(treeId).getRootNode());

  return true;
}


/**
 *
 */
bool XmlLingDefFormatter::addTreeNode(xmlNodePtr nodeParentNode,
				      LingDef &lingDef,
				      LingDef::Tree::Node &parent) {
  xmlXPathContextPtr xpContext = xmlXPathNewContext(nodeParentNode->doc);
  xpContext->node = nodeParentNode;

  // Go through child nodes
  {
    xmlXPathObjectPtr xpObj = xmlXPathEval((xmlChar *) "Node", xpContext);

    if (xpObj != NULL) {
      xmlNodeSetPtr nodeSet = xpObj->nodesetval;
      if (nodeSet != NULL && nodeSet->nodeNr > 0) {
	{for (int i = 0; i < nodeSet->nodeNr; i++) {
	  if (langOk(nodeSet->nodeTab[i], lingDef)) {
	    xmlNodePtr nodeNode = nodeSet->nodeTab[i];

	    string nodeName;
	    {
	      xmlChar *tmp = xmlGetProp(nodeNode, (xmlChar *) "name");
	      if (tmp != NULL) {
		nodeName = (char *) tmp;
		xmlFree(tmp);
	      }
	    }

	    addTreeNode(nodeNode, lingDef, parent.createChildNode(nodeName));
	  }
	}}
      }
      xmlXPathFreeObject(xpObj);
    }
  }

  xmlXPathFreeContext(xpContext);

  return true;
}

/**
 *
 */
string XmlLingDefFormatter::output(const LingDef &) const {
  return "";
}

/**
 *
 */
void XmlLingDefFormatter::output(const LingDef &, ostream &) const {
}
