#ifndef _LINGUISTICDEFINITION_WORDPATTERN_H_
#define _LINGUISTICDEFINITION_WORDPATTERN_H_

#include <string>
#include <vector>
#include <set>

#include "LingDef.h"
#include "LingFeatures.h"
#include "LingFeaturesSet.h"

//TODO: Create matcher classes!

namespace LinguisticDefinition {

  /**
   * A pattern to match one word, or rather one LingFeatures object.
   *
   * Since a LingFeaturesSet object can be considered as an extended
   * LingFeatures object, it is possible to match that as well.
   *
   * For example, to express <i>A noun <b>and</b> either feminine gender
   * <b>and</b> nominate case, <b>or</b> masculine gender <b>and</b> gentive
   * case</i> the following WordPattern may be created:
   *
   * <pre><code>
   *  WordPattern pattern;
   *
   *  WordPattern::GroupElement &rootElement =
   *    pattern.getModifiableRootElement();
   *
   *  WordPattern::GroupElement &andGroup1 = rootElement.createAnd();
   *
   *  WordPattern::LeafElement &nounLeaf = and1.createLeaf();
   *  nounLeaf.setPosDef(*nounDef); // The nounDef object is retrieved from
   *                                // some LingDef somewhere
   *
   *  WordPattern::GroupElement &orGroup = andGroup1.createOr();
   *
   *  WordPattern::GroupElement &andGroup2 = orGroup.createAnd();
   *
   *  WordPattern::LeafElement &feminineLeaf = andGroup2.createLeaf();
   *  feminineLeaf.setFeature(*feminineDef);
   *
   *  WordPattern::LeafElement &nominativeLeaf = andGroup2.createLeaf();
   *  nominativeLeaf.setFeature(*nominativeDef);
   *
   *  WordPattern::GroupElement &andGroup3 = orGroup.createAnd();
   *
   *  WordPattern::LeafElement &masculineLeaf = andGroup3.createLeaf();
   *  masculineLeaf.setFeature(*masculineDef);
   *
   *  WordPattern::LeafElement &genitiveLeaf = andGroup3.createLeaf();
   *  genitiveLeaf.setFeature(*genitiveDef);
   * </code></pre>
   *
   * Sub classes of WordPatternFormatter may do this for you, using an external
   * source of some sort.
   */
  class WordPattern {
  public:

    /**
     *
     */
    WordPattern();

    /**
     *
     */
    ~WordPattern();

    /**
     *
     */
    WordPattern(const WordPattern &);

    /**
     *
     */
    bool matches(const LingFeatures &) const;

    /**
     *
     */
    bool matches(const LingFeaturesSet &) const;

    /**
     *
     */
    LingFeaturesSet reduce(const LingFeaturesSet &) const;

    /**
     *
     */
    void generate(std::vector<const LingFeatures *> &) const;

    class Element;
    class OrGroupElement;
    class AndGroupElement;
    class LeafElement;

    /**
     *
     */
    OrGroupElement &getModifiableRootElement();

    /**
     *
     */
    const OrGroupElement &getRootElement() const;

    /**
     */
    class Element {
    public:

      /**
       *
       */
      Element(const WordPattern &);

      /**
       *
       */
      virtual ~Element();

      /**
       *
       */
      Element(const Element &);

      /**
       *
       */
      virtual Element *clone() const = 0;

      /**
       *
       */
      const WordPattern &getElement() const;

      /**
       *
       */
      void setNegative(bool = true);

      /**
       *
       */
      bool isNegative() const;

      /**
       *
       */
      virtual bool matches(const LingFeatures &) const = 0;

      /**
       *
       */
      virtual void generate(std::vector<const LingFeatures *> &,
			    LingFeatures *current) const = 0;

    private:
      const WordPattern &d_element;
      bool d_negative;
    };

    /**
     *
     */
    class GroupElement : public Element {
    public:

      /**
       *
       */
      GroupElement(const WordPattern &);

      /**
       *
       */
      ~GroupElement();

      /**
       *
       */
      GroupElement(const GroupElement &);

      /**
       *
       */
      OrGroupElement &createOr();

      /**
       *
       */
      AndGroupElement &createAnd();

      /**
       *
       */
      LeafElement &createLeaf();

      /**
       *
       */
      typedef std::vector<const Element *> ElementList;

      /**
       *
       */
      typedef ElementList::const_iterator ElementIterator;

      /**
       *
       */
      ElementIterator elementsBegin() const;

      /**
       *
       */
      ElementIterator elementsEnd() const;

    private:
      ElementList d_elements;
    };

    /**
     *
     */
    class OrGroupElement : public GroupElement {
    public:

      /**
       *
       */
      OrGroupElement(const WordPattern &);

      /**
       *
       */
      ~OrGroupElement();

      /**
       *
       */
      OrGroupElement(const OrGroupElement &);

      /**
       *
       */
      Element *clone() const;

      /**
       *
       */
      bool matches(const LingFeatures &) const;

      /**
       *
       */
      void generate(std::vector<const LingFeatures *> &,
		    LingFeatures *current) const;

    private:
    };

    /**
     *
     */
    class AndGroupElement : public GroupElement {
    public:

      /**
       *
       */
      AndGroupElement(const WordPattern &);

      /**
       *
       */
      ~AndGroupElement();

      /**
       *
       */
      AndGroupElement(const AndGroupElement &);

      /**
       *
       */
      Element *clone() const;

      /**
       *
       */
      bool matches(const LingFeatures &) const;

      /**
       *
       */
      void generate(std::vector<const LingFeatures *> &,
		    LingFeatures *current) const;

    private:
    };

    /**
     *
     */
    class LeafElement : public Element {
    public:

      /**
       *
       */
      LeafElement(const WordPattern &);

      /**
       *
       */
      ~LeafElement();

      /**
       *
       */
      LeafElement(const LeafElement &);

      /**
       *
       */
      Element *clone() const;

      /**
       *
       */
      const LingDef::Pos *getPosDef() const;

      /**
       *
       */
      void setPosDef(const LingDef::Pos &);

      /**
       *
       */
      void setForm(const std::string &);

      /**
       *
       */
      void setLemma(const std::string &);

      /**
       *
       */
      void setFeature(const LingDef::Feature &);

      /**
       *
       */
      void setFeature(const LingDef::Feature &enumFeature,
		      const LingDef::Feature &enumValueFeature);

      /**
       *
       */
      bool matches(const LingFeatures &) const;

      /**
       *
       */
      void generate(std::vector<const LingFeatures *> &,
		    LingFeatures *current) const;

    private:
      const LingDef::Pos *d_pos;
      const LingDef::Feature *d_feature;
      std::string d_form;
      std::string d_lemma;
    };

    /**
     *
     */
    class Reference {
    public:

      /**
       *
       */
      Reference(const std::string &featureName,
		int sourceElementIndex,
		const std::string &sourceFeatureName);

      /**
       *
       */
      const std::string &getFeatureName() const;

      /**
       *
       */
      int getSourceElementIndex() const;

      /**
       *
       */
      const std::string &getSourceFeatureName() const;

    private:
      std::string d_featureName;
      int d_sourceElementIndex;
      std::string d_sourceFeatureName;
    };

    /**
     *
     */
    //TODO: references should be part of the and/or structure, but let's start
    //      by doing it the easy way
    Reference &createReference(const std::string &featureName,
			       int sourceElementIndex,
			       const std::string &sourceFeatureName);

    /**
     *
     */
    typedef std::vector<const Reference *> ReferenceList;

    /**
     *
     */
    typedef ReferenceList::const_iterator ReferenceIterator;

    /**
     *
     */
    ReferenceIterator referencesBegin() const;

    /**
     *
     */
    ReferenceIterator referencesEnd() const;

  private:
    OrGroupElement d_rootElement;
    ReferenceList d_references;
  };

}

#endif //_LINGUISTICDEFINITION_WORDPATTERN_H_
