#ifndef _LINGUISTICDEFINITION_LINGFEATURESSET_H_
#define _LINGUISTICDEFINITION_LINGFEATURESSET_H_

#include "LingDef.h"
#include "LingFeatures.h"

#include <string>
#include <set>

namespace LinguisticDefinition {

  /**
   * Describes a set of LingFeatures objects, useful for representing
   * ambiguities.
   *
   * The use and the principle is very close to that of LingFeatures, but there
   * are some important differences.
   *
   * In its most simple form, each LingFeatures object is added explictly to
   * the set:
   *
   * <pre><code>
   *  LingFeatureSet featuresSet;
   *
   *  LingFeatures nounSingMasc(*nounPos);
   *  nounSingMasc += "singular";
   *  nounSingMasc += "masculine";
   *  featuresSet.add(nounSingMasc);
   *
   *  LingFeatures nounPlurMasc(*nounPos);
   *  nounPlurMasc += "plural";
   *  nounplurMasc += "masculine";
   *  featuresSet.add(nounPlurMasc);
   *
   *  featuresSet.has("singular");  // Returns true
   *  featuresSet.has("plural");    // Returns true
   *  featuresSet.has("masculine"); // Returns true
   *  featuresSet.has("feminine");  // Returns false
   * </code></pre>
   *
   * To test more complex expressions, the class MatchExpression may be used.
   *
   * LingFeatures objects may also be added dynamically as features are added
   * to the set:
   *
   * <pre><code>
   *  LingFeatureSet featuresSet;
   *
   *  LingFeatures nounSing(*nounPos);
   *  nounSing += "singular";
   *  featuresSet.add(nounSing);
   *  
   *  // Here, featuresSet only contains one element:
   *  //  NOUN:+singular
   *
   *  featuresSet += "plural";
   *
   *  // Now, featuresSet contains two elements
   *  //  NOUN:+singular
   *  //  NOUN:+plural
   *
   *  featuresSet += "masculine";
   *  featuresSet += "feminine";
   *
   *  // Now, featuresSet contains eight elements
   *  //  NOUN:+singular
   *  //  NOUN:+plural
   *  //  NOUN:+singular+masculine
   *  //  NOUN:+plural+masculine
   *  //  NOUN:+singular+feminine
   *  //  NOUN:+plural+feminine
   *
   * </code></pre>
   * 
   */
  class LingFeaturesSet {
  public:

    /**
     *
     */
    LingFeaturesSet();

    /**
     *
     */
    ~LingFeaturesSet();

    /**
     * The LingFeatures object is copied
     */
    void add(const LingFeatures &);

    /**
     *
     */
    void remove(const LingFeatures *);

    /**
     *
     */
    bool isEmpty() const;

    /**
     *
     */
    void set(const std::string &feature);

    /**
     *
     */
    void set(const LingDef::Feature &);

    /**
     *
     */
    bool has(const std::string &feature) const;

    /**
     *
     */
    bool has(const LingDef::Feature &) const;

    /**
     *
     */
    void setEnum(const std::string &enumFeature,
		 const std::string &valueFeature);

    /**
     *
     */
    void setEnum(const LingDef::Feature &enumFeature,
		 const LingDef::Feature &valueFeature);

    /**
     *
     */
    void getEnumValues(const std::string &enumFeature,
		       std::set<const LingDef::Feature *> &) const;

    /**
     *
     */
    void getEnumValues(const LingDef::Feature &enumFeature,
		       std::set<const LingDef::Feature *> &) const;

    /**
     *
     */
    typedef std::set<LingFeatures *> ElementSet;

    /**
     *
     */
    typedef ElementSet::const_iterator ElementIterator;

    /**
     *
     */
    ElementIterator elementsBegin() const;

    /**
     *
     */
    ElementIterator elementsEnd() const;

  private:
    const LingDef *d_lingDef;
    const LingDef::Pos *d_posDef;
    bool d_sharedPosDef;

    ElementSet d_elements;
    ElementSet d_removedElements;
  };

}

#endif //_LINGUISTICDEFINITION_LINGFEATURESSET_H_
