#ifndef _LINGUISTICDEFINITION_LINGFEATURES_H_
#define _LINGUISTICDEFINITION_LINGFEATURES_H_

#include <string>
#include <vector>
#include <map>
#include <set>
#include <sstream>
#include <iostream>

#include "LingDef.h"

//TODO: replace 'reference' with 'int value'/'string value'?
//TODO: reference class?
//TODO: "internal" references (for derivation information, for example)

namespace LinguisticDefinition {

  /**
   * Represents an instance of a part-of-speech (pos) definition in a LingDef
   * object.
   *
   * The only mandatory information needed to create this instance is the pos.
   *
   * Features can be added, removed and checked:
   *
   * <pre><code>
   *  features.set("plural");
   *  features.set("masculine");
   *
   *  features.has("number"); // Returns true, if 'plural' is an enumeration
   *                          // value of 'number'
   *
   *  features.unset("plural");
   *
   *  features.has("number"); // Returns false
   *
   *  features.unset("gender");
   *
   *  features.has("masculine"); // Returns false
   * </code></pre>
   *
   * This assuming that the pos for which the object <code>features</code> is
   * created has an enumeration <code>number</code> containing
   * <code>singular</code> and <code>plural</code>, and another called
   * <code>gender</code> containing <code>masculine</code> and
   * <code>feminine</code>.
   *
   * There are also two fix string properties for a LingFeatures object, lemma
   * and form. Having them among the linguistic properties allows for a good
   * abstraction of a word being a LingFeatures object.
   *
   */
  class LingFeatures {
    friend class LingDef;
  public:

    /**
     * @param posDef The pos definition to instantiate. Note that the pos
     *               definition contains the reference to a LingDef object,
     *               so there is no need to specify that here.
     */
    LingFeatures(const LingDef::Pos &posDef);

    /**
     *
     */
    ~LingFeatures();

    /**
     * The pos definition must be set using setPosDef() before
     * using the object.
     */
    LingFeatures(const LingDef &lingDef);

    /**
     * TODO
     */
    class Exception {
    public:

      /**
       *
       */
      Exception(const std::string &);

      /**
       *
       */
      const std::string &getMessage() const;

    private:
      std::string d_message;
    };

    /**
     * Returns the LingDef object for which this object is created.
     */
    const LingDef *getLingDef() const;

    /**
     * Returns the pos definition for which this LingFeatures object is
     * created.
     * May be NULL for global features (hmm... they aren't really
     * global).
     */
    const LingDef::Pos *getPosDef() const;

    /**
     * Sets the pos definition for which this object will be defined. This
     * should only be done on freshly created objects, since otherwise the
     * old features will come from another pos, which does not make any sense.
     */
    void setPosDef(const LingDef::Pos &);

    /**
     *
     */
    const std::string &getLemma() const;

    /**
     *
     */
    void setLemma(const std::string &);

    /**
     *
     */
    const std::string &getForm() const;

    /**
     *
     */
    void setForm(const std::string &);

    /**
     * Sets the feature with the name <code>feature</code>. This is valid if
     * <code>feature</code> is a feature defined for the pos for which this
     * LingFeatures object is created, and
     * <ul>
     *  <li>a free boolean, or</li>
     *  <li>a value in an enumeration, or</li>
     *  <li>a value in a virtual tree used by this pos, or</li>
     *  <li>an enumeration which has a default value</li>
     * </ul>
     */
    void set(const std::string &feature);

    /**
     *
     */
    void setNegative(const std::string &feature);

    /**
     * Unsets the feature with the name <code>feature</code>. All types may be
     * unset. If an enumeration is unset, then the enumeration value will be
     * unset.
     */
    void unset(const std::string &feature);

    /**
     * Returns wether or not the feature with the name <code>feature</code> is
     * set. All types may be checked. If an enumeration has a value set, then
     * the LingFields object "has" the enumeration feature as well.
     */
    bool has(const std::string &feature) const;

    /**
     *
     */
    bool hasNegative(const std::string &feature) const;

    /**
     *
     */
    bool isDefined(const std::string &feature) const;

    /**
     * Same as set(string), except the lookup step in the parent LingDef is
     * skipped.
     */
    void set(const LingDef::Feature &);

    /**
     *
     */
    void setNegative(const LingDef::Feature &);

    /**
     * Same as unset(string), except the lookup step in the parent LingDef is
     * skipped.
     */
    void unset(const LingDef::Feature &);

    /**
     * Same as has(string), except the lookup step in the parent LingDef is
     * skipped.
     */
    bool has(const LingDef::Feature &) const;

    /**
     * Same as hasNegative(string), except the lookup step in the parent
     * LingDef is skipped.
     */
    bool hasNegative(const LingDef::Feature &) const;

    /**
     *
     */
    bool isDefined(const LingDef::Feature &) const;

    /**
     * Same as set(std::string &).
     */
    const LingFeatures &operator+=(const std::string &);

    /**
     * Same as set(LingDef::Feature &).
     */
    const LingFeatures &operator+=(const LingDef::Feature &);

    /**
     * Same as setNegative(std::string &).
     */
    const LingFeatures &operator-=(const std::string &);

    /**
     * Same as setNegative(LingDef::Feature &).
     */
    const LingFeatures &operator-=(const LingDef::Feature &);

    /**
     * Sets the value of the reference feature by the name of
     * <code>feature</code>.
     *
     * If so defined, a reference feature may have several values. This
     * function will then add a reference value, otherwise it will replace the
     * existing value.
     */
    void setReference(const std::string &feature, int);

    /**
     * Returns the value of the reference feature by the name of
     * <code>feature</code>.
     *
     * If the reference feature has several values, this will return one of
     * these in an unspecified manner. Testing of references with several
     * values is done easily with hasReference().
     */
    int getReference(const std::string &feature) const;

    /**
     * Returns a list containing the values for the reference feature by the
     * name of <code>feature</code>. If the reference feature only allows one
     * value, the list will only contain one element.
     */
    void getReferenceList(const std::string &feature,
			  std::vector<int> &) const;

    /**
     * Returns the value of the reference feature by the name of
     * <code>feature</code> is <code>referenceValue</code>. This may be used
     * when the reference feature has several values, instead of testing the
     * result of getReference().
     *
     * If the value of the reference is unimportant, and only the existence of
     * a value for the feature is interesting, then the general has() function
     * may be used.
     */
    bool hasReference(const std::string &feature, int referenceValue) const;

    /**
     *
     */
    void setReference(const LingDef::Feature &, int);

    /**
     *
     */
    int getReference(const LingDef::Feature &) const;

    /**
     *
     */
    void getReferenceList(const LingDef::Feature &, std::vector<int> &) const;

    /**
     *
     */
    bool hasReference(const LingDef::Feature &, int) const;

    /**
     *
     */
    void setEnum(const std::string &enumFeature,
		 const std::string &valueFeature);

    /**
     *
     */
    void setEnum(const LingDef::Feature &enumFeature,
		 const LingDef::Feature &valueFeature);

    /**
     *
     */
    const LingDef::Feature *getEnumValue(const std::string &) const;

    /**
     *
     */
    const LingDef::Feature *getEnumValue(const LingDef::Feature &) const;

    /**
     *
     */
    void getEnumValues(const std::string &,
		       std::set<const LingDef::Feature *> &) const;

    /**
     *
     */
    void getEnumValues(const LingDef::Feature &,
		       std::set<const LingDef::Feature *> &) const;

    /**
     * @return 0 for false, 1 for only this value, 2 for this among others
     */
    int hasEnumValue(const std::string &enumValueName) const;

    /**
     * @return 0 for false, 1 for only this value, 2 for this among others
     */
    int hasEnumValue(const LingDef::Feature &enumValueFeature) const;

    /**
     * This LingFeatures object covers another LingFeatures object if all the
     * features in the other LingFeatures object are present in this
     * LingFeatures object.
     *
     * Example:
     * <pre><code>
     *  a.set("plural");
     *  a.set("masculine");
     *
     *  b.set("plural");
     *
     *  c.set("plural");
     *  c.set("feminine");
     *
     *  a.covers(a); // true
     *  a.covers(b); // true
     *  b.covers(a); // false
     *  a.covers(c); // false
     *  c.covers(b); // true
     * </code></pre>
     *
     */
    bool covers(const LingFeatures &) const;

    /**
     * Sets and unsets features as defined by a string on the form
     *
     * <pre><code>
     *  +definite+gender=masculine-number+modified_by=$2
     * </code></pre>
     *
     * which would correspond to
     *
     * <pre><code>
     *  features.set("definite");
     *  features.set("masculine");
     *  features.unset("number");
     *  features.setReference("modified_by", 2);
     * </code></pre>
     *
     */
    void execute(const std::string &); //TODO: does this belong in this class?

    /**
     *
     */
    void setDefaults();

    /**
     *
     */
    void setDefaultForEnum(const LingDef::Feature &enumFeature);

    /**
     *
     */
    typedef std::set<const LingDef::Feature *> FeaturesList;

    /**
     *
     */
    typedef FeaturesList::const_iterator FeatureIterator;

    /**
     *
     */
    FeatureIterator featuresBegin() const;

    /**
     *
     */
    FeatureIterator featuresEnd() const;

    /**
     *
     */
    typedef std::multimap<const LingDef::Feature *,
			  const LingDef::Feature *> EnumMap;

    /**
     *
     */
    typedef std::multimap<const LingDef::Feature *, int> ReferenceMap;

  protected:

    /**
     * Dangerous!
     */
    LingFeatures();

  private:

    /**
     *
     */
    LingFeatures(const LingDef &, const std::string &expression);

    /**
     *
     */
    void setSub(const LingDef::Feature &);

    /**
     *
     */
    void unsetSub(const LingDef::Feature &);

  private:
    const LingDef *d_lingDef;
    const LingDef::Pos *d_posDef;
    FeaturesList d_features;
    FeaturesList d_negativeFeatures;
    EnumMap d_enums;
    ReferenceMap d_references;

    std::string d_lemma;
    std::string d_form;
  };

}

#endif //_LINGUISTICDEFINITION_LINGFEATURES_H_
