#include "LinguisticDefinition/LingDef.h"

using namespace std;
using namespace LinguisticDefinition;

/**
 *
 */
LingDef::LingDef(const string &isoLanguageCode) :
  d_isoLanguageCode(isoLanguageCode) {
}

/**
 *
 */
LingDef::~LingDef() {
  {for (vector<Pos *>::iterator it = d_posDefs.begin();
	it != d_posDefs.end(); ++it) {
    delete *it;
  }}

  {for (vector<Tree *>::iterator it = d_treeDefs.begin();
	it != d_treeDefs.end(); ++it) {
    delete *it;
  }}

  {for (ConflictMap::iterator it = d_conflicts.begin();
	it != d_conflicts.end(); ++it) {
    delete (*it).second;
  }}
}

/**
 *
 */
LingDef::LingDef(const LingDef &o) :
  d_isoLanguageCode(o.d_isoLanguageCode) {

  map<const Tree *, const Tree *> oldToNewTreeMap;
  map<const Tree::Node *, const Tree::Node *> oldToNewTreeNodeMap;

  {for (vector<Tree *>::const_iterator it = o.d_treeDefs.begin();
	it != o.d_treeDefs.end(); ++it) {
    const Tree *oldTree = *it;
    Tree *newTree = new Tree(*oldTree);
    oldToNewTreeMap[oldTree] = newTree;
    addTree(newTree);
    
    {for (Tree::NodeIterator it2 = oldTree->nodesBegin();
	  it2 != oldTree->nodesEnd(); ++it2) {
      oldToNewTreeNodeMap[*it2] = newTree->getNode((*it2)->getName());
    }}
  }}
  
  map<const Pos *, Pos *> oldToNewPosMap;

  {for (vector<Pos *>::const_iterator it = o.d_posDefs.begin();
	it != o.d_posDefs.end(); ++it) {
    const Pos *oldPos = *it;
    Pos *newPos = new Pos(*oldPos);
    newPos->d_lingDef = this;
    {for (set<const Feature *>::iterator it2 = newPos->d_featureDefs.begin();
	  it2 != newPos->d_featureDefs.end(); ++it2) {
      Feature *f = (Feature *) *it2;
      f->d_lingDef = this;
      f->d_posDef = newPos;
      if (f->d_tree != NULL) {
	f->d_tree = oldToNewTreeMap[f->d_tree];
      }
      if (f->d_treeNode != NULL) {
	f->d_treeNode = oldToNewTreeNodeMap[f->d_treeNode];
      }
    }}

    oldToNewPosMap[oldPos] = newPos;
    addPos(newPos);
  }}

  {for (vector<Pos *>::iterator it = d_posDefs.begin();
	it != d_posDefs.end(); ++it) {
    Pos *pos = *it;

    // The super pos is still the old reference, update it
    if (pos->d_superPos != NULL) {
      pos->d_superPos = oldToNewPosMap[pos->d_superPos];
    }
  }}

  {for (ConflictMap::const_iterator it = o.d_conflicts.begin();
	it != o.d_conflicts.end(); ++it) {
    const Feature *oldKey = (*it).first;
    const Pos::FeatureList *oldValue = (*it).second;

    const Pos *oldKeyPos = oldKey->getPosDef();
    const Pos *newKeyPos = oldToNewPosMap[oldKeyPos];

    const Feature *newKey = newKeyPos->getFeature(oldKey->getName());
    Pos::FeatureList *newValue = new Pos::FeatureList;

    d_conflicts[newKey] = newValue;

    {for (Pos::FeatureIterator it2 = oldValue->begin();
	  it2 != oldValue->end(); ++it2) {
      const Feature *oldConflictFeature = *it2;
      const Pos *oldPos = oldConflictFeature->getPosDef();
      const Pos *newPos = oldToNewPosMap[oldPos];

      newValue->insert(newPos->getFeature(oldConflictFeature->getName()));
    }}
  }}
}

/**
 *
 */
LingDef &LingDef::operator=(const LingDef &o) {
  if (this != &o) {
    d_isoLanguageCode = o.d_isoLanguageCode;

    {for (vector<Pos *>::iterator it = d_posDefs.begin();
	  it != d_posDefs.end(); ++it) {
      delete *it;
    }}

    d_posDefs.clear();
    d_posNameMap.clear();

    {for (vector<Pos *>::const_iterator it = o.d_posDefs.begin();
	  it != o.d_posDefs.end(); ++it) {
      addPos(new Pos(**it));
    }}
  }
  return *this;
}

/**
 *
 */
const string &LingDef::getIsoLanguageCode() const {
  return d_isoLanguageCode;
}

/**
 *
 */
LingDef::Pos::Pos &LingDef::createPos(const string &name) {
  Pos *pos = new Pos(*this, name);
  addPos(pos);
  return *pos;
}

/**
 *
 */
LingDef::Pos::Pos &LingDef::Pos::createSubPos(const string &name) {
  Pos *pos = new Pos(*this, name);
  d_lingDef->addPos(pos);
  return *pos;
}

/**
 *
 */
LingDef::Pos::Pos &LingDef::createVirtualPos() {
  return createPos("");
}

/**
 *
 */
LingDef::Pos::Pos &LingDef::Pos::createVirtualSubPos() {
  return createSubPos("");
}

/**
 *
 */
LingDef::Pos::Pos(LingDef &lingDef,
		  const string &name) :
  d_lingDef(&lingDef),
  d_superPos(NULL),
  d_name(name) {
}

/**
 *
 */
LingDef::Pos::Pos(Pos &superPos,
		  const string &name) :
  d_lingDef(superPos.d_lingDef),
  d_superPos(&superPos),
  d_name(name) {
}

/**
 *
 */
LingDef::Pos::Pos(const Pos &o) :
  d_lingDef(o.d_lingDef),
  d_superPos(o.d_superPos), // Set by copy constructor in LingDef
  d_name(o.d_name),
  d_shortName(o.d_shortName),
  d_type(o.d_type),
  d_note(o.d_note) {

  if (this != &o) {
    vector<Feature *> newFeatures;
    map<Feature *, const Feature *> newToOldMap;
    map<const Feature *, Feature *> oldToNewMap;
    {for (set<const Feature *>::const_iterator
	    it = o.d_featureDefs.begin();
	  it != o.d_featureDefs.end(); ++it) {
      const Feature *oldFeature = *it;
      Feature *newFeature = new Feature(*oldFeature);

      addFeature(newFeature);
      newToOldMap[newFeature] = oldFeature;
      oldToNewMap[*it] = newFeature;
      newFeatures.push_back(newFeature);
    }}

    {for (vector<Feature *>::iterator it = newFeatures.begin();
	  it != newFeatures.end(); ++it) {
      Feature *newFeature = *it;

      if (newFeature->d_parentEnum != NULL) {
	newFeature->d_parentEnum = oldToNewMap[newFeature->d_parentEnum];
      }

      const Feature *oldFeature = newToOldMap[newFeature];
      newFeature->d_enumChildren.clear();
      {for (set<Feature *>::const_iterator
	      it2 = oldFeature->d_enumChildren.begin();
	    it2 != oldFeature->d_enumChildren.end(); ++it2) {
	newFeature->d_enumChildren.insert(oldToNewMap[*it2]);
      }}
    }}
  }
}

/**
 *
 */
LingDef::Pos &LingDef::Pos::operator=(const Pos &o) {
  d_lingDef = o.d_lingDef;
  d_superPos = o.d_superPos;
  d_name = o.d_name;
  d_shortName = o.d_shortName;
  d_type = o.d_type;
  d_note = o.d_note;
  if (this != &o) {

    {for (set<const Feature *>::iterator it = d_featureDefs.begin();
	  it != d_featureDefs.end(); ++it) {
      delete *it;
    }}

    d_featureDefs.clear();
    d_featureNameMap.clear();
    d_defaultFeatures.clear();

    {for (set<const Feature *>::iterator it = o.d_featureDefs.begin();
	  it != o.d_featureDefs.end(); ++it) {
      addFeature(new Feature(**it));
    }}
  }

  return *this;
}

/**
 *
 */
LingDef::Pos::~Pos() {
  {for (set<const Feature *>::iterator it = d_featureDefs.begin();
	it != d_featureDefs.end(); ++it) {
    delete *it;
  }}
}

/**
 *
 */
const string &LingDef::Pos::getName() const {
  return d_name;
}

/**
 *
 */
void LingDef::Pos::setShortName(const string &val) {
  d_shortName = val;
  if (d_lingDef != NULL && val != "") {
    d_lingDef->d_posShortNameMap[val] = this;
  }
}

/**
 *
 */
const string &LingDef::Pos::getShortName() const {
  return d_shortName;
}

/**
 *
 */
const LingDef *LingDef::Pos::getLingDef() const {
  return d_lingDef;
}

/**
 *
 */
const LingDef::Pos *LingDef::Pos::getSuperPos() const {
  return d_superPos;
}

/**
 *
 */
void LingDef::addPos(LingDef::Pos *pos) {
  d_posDefs.push_back(pos);
  if (pos->getName() != "") {
    d_posNameMap[pos->getName()] = pos;
  }
}

/**
 *
 */
const LingDef::Pos *LingDef::getPos(const string &id) const {
  map<string, Pos *>::const_iterator findIt = d_posNameMap.find(id);
  if (findIt == d_posNameMap.end()) {
    map<string, Pos *>::const_iterator findIt2 = d_posShortNameMap.find(id);
    if (findIt2 == d_posShortNameMap.end()) {
      return NULL;
    }
    return (*findIt2).second;
  }
  return (*findIt).second;
}

/**
 *
 */
void LingDef::Pos::setType(LingDef::Pos::Type type) {
  d_type = type;
}

/**
 *
 */
LingDef::Pos::Type LingDef::Pos::getType() const {
  return d_type;
}

/**
 *
 */
void LingDef::Pos::setNote(const string &val) {
  d_note = val;
}

/**
 *
 */
LingDef::Feature &LingDef::Pos::createFeature(const string &name,
					      LingDef::Feature::Domain domain,
					      LingDef::Feature::Type type) {
  Feature *feature = new Feature(*this, name, domain, type);
  addFeature(feature);
  return *feature;
}

/**
 *
 */
LingDef::Feature &
LingDef::Feature::createEnumValueFeature(const string &name) {
  bool firstValue = d_enumChildren.empty();

  Feature *feature = new Feature(*this, name);

  d_posDef->addFeature(feature);

  if (firstValue && !allowNoValue()) {
    feature->setDefault(true);
  }

  return *feature;
}

/**
 *
 */
void LingDef::Pos::addFeature(LingDef::Feature *feature) {
  d_featureDefs.insert(feature);
  d_featureNameMap[feature->getName()] = feature;
}

/**
 *
 */
LingDef::Pos::FeatureIterator LingDef::Pos::featuresBegin() const {
  return d_featureDefs.begin();
}

/**
 *
 */
LingDef::Pos::FeatureIterator LingDef::Pos::featuresEnd() const {
  return d_featureDefs.end();
}

/**
 *
 */
LingDef::Pos::FeatureIterator LingDef::Pos::defaultFeaturesBegin() const {
  return d_defaultFeatures.begin();
}

/**
 *
 */
LingDef::Pos::FeatureIterator LingDef::Pos::defaultFeaturesEnd() const {
  return d_defaultFeatures.end();
}

/**
 *
 */
void LingDef::Feature::addConflict(const LingDef::Feature &feature) {
  if (d_lingDef != NULL) {
    d_lingDef->addConflict(*this, feature);
  }
}

/**
 *
 */
void LingDef::addConflict(const LingDef::Feature &featureA,
			  const LingDef::Feature &featureB) {

  // One way
  {
    Pos::FeatureList *conflictSet = NULL;
    ConflictMap::const_iterator findIt = d_conflicts.find(&featureA);
    if (findIt == d_conflicts.end()) {
      conflictSet = new Pos::FeatureList;
      d_conflicts[&featureA] = conflictSet;
    } else {
      conflictSet = (*findIt).second;
    }
    conflictSet->insert(&featureB);
  }

  // The other way
  {
    Pos::FeatureList *conflictSet = NULL;
    ConflictMap::const_iterator findIt = d_conflicts.find(&featureB);
    if (findIt == d_conflicts.end()) {
      conflictSet = new Pos::FeatureList;
      d_conflicts[&featureB] = conflictSet;
    } else {
      conflictSet = (*findIt).second;
    }
    conflictSet->insert(&featureA);
  }
}

/**
 *
 */
const LingDef::Pos::FeatureList *
LingDef::getConflicts(const Feature &feature) const {
  ConflictMap::const_iterator findIt = d_conflicts.find(&feature);
  if (findIt == d_conflicts.end()) {
    return NULL;
  }
  return (*findIt).second;
}

/**
 *
 */
LingDef::Feature::Feature(LingDef::Pos &posDef,
			  const string &name,
			  LingDef::Feature::Domain domain,
			  LingDef::Feature::Type type) :
  d_lingDef(posDef.d_lingDef),
  d_posDef(&posDef),
  d_name(name),
  d_domain(domain),
  d_type(type),
  d_parentEnum(NULL),
  d_allowSeveralValues(false),
  d_allowNoValue(true),
  d_isDefault(false),
  d_tree(NULL),
  d_treeNode(NULL) {
}

/**
 * For enum values
 */
LingDef::Feature::Feature(LingDef::Feature &parentEnum,
			  const string &name) :
  d_lingDef(parentEnum.d_lingDef),
  d_posDef(parentEnum.d_posDef),
  d_name(name),
  d_domain(parentEnum.d_domain),
  d_type(BOOLEAN),
  d_parentEnum(&parentEnum),
  d_allowSeveralValues(false),
  d_allowNoValue(true),
  d_isDefault(false),
  d_tree(NULL),
  d_treeNode(NULL) {

  d_parentEnum->d_enumChildren.insert(this);
}

/**
 *
 */
LingDef::Feature::~Feature() {
}

/**
 *
 */
const string &LingDef::Feature::getName() const {
  return d_name;
}


/**
 *
 */
const LingDef *LingDef::Feature::getLingDef() const {
  return d_lingDef;
}

/**
 *
 */
const LingDef::Pos *LingDef::Feature::getPosDef() const {
  return d_posDef;
}

/**
 *
 */
const LingDef::Feature *LingDef::Pos::getFeature(const string &name)
  const {

  //TODO? Put parent features in the local map instead?

  map<string, const Feature *>::const_iterator
    findIt = d_featureNameMap.find(name);
  if (findIt == d_featureNameMap.end()) {
    const Pos *superPos = getSuperPos();
    if (superPos != NULL) {
      const Feature *superPosFeature = superPos->getFeature(name);
      if (superPosFeature == NULL) {
	map<string, const Feature *>::const_iterator findIt2 =
	  d_featureShortNameMap.find(name);
	if (findIt2 == d_featureShortNameMap.end()) {
	  return NULL;
	}
	return (*findIt2).second;
      }
      return superPosFeature;
    }
    return NULL;
  }

  return (*findIt).second;
}

/**
 *
 */
LingDef::Feature::Domain LingDef::Feature::getDomain() const {
  if (d_parentEnum != NULL) {
    return d_parentEnum->getDomain();
  }
  return d_domain;
}

/**
 *
 */
LingDef::Feature::Type LingDef::Feature::getType() const {
  return d_type;
}

/**
 *
 */
const LingDef::Feature *LingDef::Feature::getParentEnum() const {
  return d_parentEnum;
}

/**
 *
 */
bool LingDef::Feature::
isIn(const set<const LingDef::Feature *> &featureSet) const {
  if (featureSet.find(this) != featureSet.end()) {
    return true;
  }

  // Is this feature a vtree feature?
  const Tree::Node *treeNode = getTreeNode();
  if (treeNode != NULL) {
    {for (set<const LingDef::Feature *>::const_iterator
	    it = featureSet.begin();
	  it != featureSet.end(); ++it) {
      // Is the other feature is also a vtree feature?
      const Tree::Node *otherTreeNode = (*it)->getTreeNode();
      if (otherTreeNode != NULL) {
	// If this feature is an ancestor of the other feature, then it is ok
	if (treeNode->isAncestorOf(*otherTreeNode)) {
	  return true;
	}
      }
    }}
  }

  // Normal case
  return false;
}

/**
 *
 */
void LingDef::Feature::setAllowSeveralValues(bool val) {
  d_allowSeveralValues = val;
}

/**
 *
 */
bool LingDef::Feature::allowSeveralValues() const {
  return d_allowSeveralValues;
}

/**
 *
 */
void LingDef::Feature::setAllowNoValue(bool val) {
  d_allowNoValue = val;
}

/**
 *
 */
bool LingDef::Feature::allowNoValue() const {
  return d_allowNoValue;
}

/**
 *
 */
void LingDef::Feature::setDefault(bool val) {
  setDefaultSub(val, true);
}

/**
 *
 */
void LingDef::Feature::setDefaultSub(bool val, bool goThroughSiblings) {
  if (val != d_isDefault) {
    if (d_posDef != NULL) {
      if (val) {
	d_posDef->d_defaultFeatures.insert(this);

	if (goThroughSiblings) {
	  const Feature *parentEnum = getParentEnum();
	  if (parentEnum != NULL) {
	    // Go through enum and remove other defaults
	    {for (set<Feature *>::iterator it =
		    parentEnum->d_enumChildren.begin();
		  it != parentEnum->d_enumChildren.end(); ++it) {
	      (*it)->setDefaultSub(false, false);
	    }}
	  }
	}

      } else {
	d_posDef->d_defaultFeatures.erase(this);

	if (goThroughSiblings && !allowNoValue()) {
	  //TODO: must set first value in enum, but we have a set...
	}
      }
    }
    d_isDefault = val;
  }
}

/**
 *
 */
bool LingDef::Feature::isDefault() const {
  return d_isDefault;
}

/**
 *
 */
void LingDef::Feature::setShortName(const string &val) {
  d_shortName = val;
  if (d_posDef != NULL && val != "") {
    d_posDef->d_featureShortNameMap[val] = this;
  }
}

/**
 *
 */
const string &LingDef::Feature::getShortName() const {
  return d_shortName;
}

/**
 *
 */
const LingDef::Tree *LingDef::Feature::getTree() const {
  return d_tree;
}

/**
 * The tree has to be completely constructed before doing this, its structure
 * is mirrored.
 */
void LingDef::Feature::setTree(const LingDef::Tree &tree) {
  d_tree = &tree;
  {for (Tree::NodeIterator it = tree.nodesBegin();
	it != tree.nodesEnd(); ++it) {
    const Tree::Node *node = *it;
    Feature &nodeFeature = createEnumValueFeature(node->getName());
    nodeFeature.d_tree = &tree;
    nodeFeature.d_treeNode = node;
  }}
}

/**
 *
 */
const LingDef::Tree::Node *LingDef::Feature::getTreeNode() const {
  return d_treeNode;
}

/**
 *
 */
LingDef::Tree &LingDef::createTree(const string &name) {
  Tree *tree = new Tree(name);
  addTree(tree);
  return *tree;
}

/**
 *
 */
const LingDef::Tree *LingDef::getTree(const string &name) const {
  map<string, Tree *>::const_iterator findIt = d_treeNameMap.find(name);
  if (findIt == d_treeNameMap.end()) {
    return NULL;
  }
  return (*findIt).second;
}

/**
 *
 */
void LingDef::addTree(LingDef::Tree *tree) {
  d_treeDefs.push_back(tree);
  if (tree->getName() != "") {
    d_treeNameMap[tree->getName()] = tree;
  }
}

/**
 *
 */
LingDef::Tree::Tree(const Tree &o) :
  d_name(o.d_name),
  d_rootNode(o.d_rootNode) {

  map<const Node *, Node *> oldToNewMap;
  {for (NodeIterator it = o.nodesBegin(); it != o.nodesEnd(); ++it) {
    const Node *oldNode = *it;
    Node *newNode = new Node(*oldNode);
    newNode->d_parentTree = this;
    oldToNewMap[oldNode] = newNode;
    d_nodes.push_back(newNode);
    d_nodeNameMap[newNode->getName()] = newNode;
  }}

  // The root node does not contain any real information, no need to copy

  {for (multimap<const Node *, const Node *>::const_iterator
	  it = o.d_ancestorsMap.begin();
	it != o.d_ancestorsMap.end(); ++it) {
    pair<const Node *, const Node *> p(oldToNewMap[(*it).first],
				       oldToNewMap[(*it).second]);
    d_ancestorsMap.insert(p);
    d_grandChildAncestorPairSet.insert(p);
  }}
}

/**
 *
 */
LingDef::Tree::Tree(const string &name) :
  d_name(name),
  d_rootNode(*this, "") {
}

/**
 *
 */
LingDef::Tree::~Tree() {
  {for (NodeIterator it = nodesBegin(); it != nodesEnd(); ++it) {
    delete *it;
  }}
}

/**
 *
 */
const string &LingDef::Tree::getName() const {
  return d_name;
}

/**
 *
 */
LingDef::Tree::Node::Node(LingDef::Tree &parentTree, const string &name) :
  d_parentTree(&parentTree),
  d_name(name) {
}

/**
 *
 */
LingDef::Tree::Node::~Node() {
}

/**
 *
 */
LingDef::Tree::Node &LingDef::Tree::Node::createChildNode(const string &name) {
  Node *node = new Node(*d_parentTree, name);
  d_parentTree->d_nodes.push_back(node);
  d_parentTree->d_nodeNameMap[name] = node;

  // This node is ancestor of the new node
  // All ancestors of this node are ancestors of the new node
  // The new node is a grand child of this node
  // The new node is a grand child of all the ancestors of this node
  {
    vector<const Node *> newAncestors;
    if (this != &d_parentTree->d_rootNode) {
      newAncestors.push_back(this);
    }
    {for (multimap<const Node *, const Node *>::iterator
	    it = d_parentTree->d_ancestorsMap.find(this);
	  it != d_parentTree->d_ancestorsMap.end() && (*it).first == this;
	  ++it) {
      const Node *ancestorNode = (*it).second;
      if (ancestorNode != &d_parentTree->d_rootNode) {
	newAncestors.push_back(ancestorNode);
      }
    }}
    {for (vector<const Node *>::iterator it = newAncestors.begin();
	  it != newAncestors.end(); ++it) {
      pair<const Node *, const Node *> p(node, *it);
      d_parentTree->d_ancestorsMap.insert(p);
      d_parentTree->d_grandChildAncestorPairSet.insert(p);
    }}
  }
  
  return *node;
}

/**
 *
 */
LingDef::Tree &LingDef::Tree::Node::getParentTree() const {
  return *d_parentTree;
}

/**
 *
 */
const string &LingDef::Tree::Node::getName() const {
  return d_name;
}

/**
 *
 */
LingDef::Tree::Node &LingDef::Tree::getRootNode() {
  return d_rootNode;
}

/**
 *
 */
const LingDef::Tree::Node *LingDef::Tree::getNode(const string &name) const {
  map<string, Node *>::const_iterator findIt = d_nodeNameMap.find(name);
  if (findIt == d_nodeNameMap.end()) {
    return NULL;
  }
  return (*findIt).second;
}

/**
 *
 */
bool LingDef::Tree::Node::isAncestorOf(const Node &grandChild) const {
  pair<const Node *, const Node *> p(&grandChild, this);
  return
    d_parentTree->d_grandChildAncestorPairSet.find(p)
    !=
    d_parentTree->d_grandChildAncestorPairSet.end();
}

/**
 *
 */
LingDef::Tree::NodeIterator LingDef::Tree::nodesBegin() const {
  return d_nodes.begin();
}

/**
 *
 */
LingDef::Tree::NodeIterator LingDef::Tree::nodesEnd() const {
  return d_nodes.end();
}
      
