#include <boost/lexical_cast.hpp>
#include <outilex/wrtn_grammar.h>
#include <outilex/xml-names.h>

using namespace std;
using namespace boost;
using namespace xmlnames;
namespace fs = boost::filesystem;

wrtn_grammar::wrtn_grammar(const fs::path & path, ling_def * ldef) 
  : lingdef(ldef), patterns(), pat_map(), axiom(0) {
    read(path);
}



int wrtn_grammar::add_pattern(const wrtn_pattern & pattern) {

  cerr << "warning: add_pattern: deep copy\n";

  int idx;
  const string & name = pattern.get_name();

  map<string, int>::iterator it = pat_map.find(name);

  if (it == pat_map.end()) {
  
    idx = patterns.size();
    patterns.resize(idx + 1);
  
    pat_map[name] = idx;

  } else { idx = it->second; }

  patterns[idx] = pattern;
  return idx;
}


int wrtn_grammar::add_pattern(const fs::path & patternpath, const std::string & name) {
  
  int idx;

  map<string, int>::iterator it = pat_map.find(name);
  
  if (it == pat_map.end()) { // new pattern
  
    idx = patterns.size();
    patterns.resize(idx + 1);

    pat_map[name] = idx;
  
  } else { // override
  
    idx = it->second;
  }

  patterns[idx].read(patternpath, lingdef);
  //cerr << "syntagm[idx].name = " << patterns[idx].get_name() << " & name = " << name << endl;
  assert(patterns[idx].get_name() == name);

  if (patterns[idx].empty()) {
    cerr << "warning: pattern " << name << " is empty\n";
    patterns[idx].add_state();
  }
  return idx;
}



void wrtn_grammar::write(const fs::path & path) const {

  xmlwriter writer(path);
  writer.set_indent(true);

  writer.start_document();
  write_xml(writer);
  writer.end_document();
}


void wrtn_grammar::read(const fs::path & path) {

  string fname = path.native_file_string();

  xmlDocPtr doc = xmlParseFile(fname.c_str());

  if (doc == NULL) {
    throw xml_parse_error("wrtn_grammar::read_XML: " + fname + " not successfully parsed");
  }

  xmlNode * node = xmlDocGetRootElement(doc);

  try {
    read_xml(node);
  } catch (exception & e) {
    cerr << "wrtn_grammar::read: error while parsing grammar " << fname << ": " << e.what() << endl;
    xmlFreeDoc(doc);
    throw;
  }
  xmlFreeDoc(doc);
}


void wrtn_grammar::write_xml(xmlwriter & writer) const {

  writer.start_element(WRTN_GRAMMAR_ELEM);

  writer.write_attribute(MAIN_ATTR, patterns[axiom].get_name());
  writer.write_attribute(SIZE_ATTR, lexical_cast<string>(patterns.size()));

  for_each(patterns.begin(), patterns.end(), XML_Writer(writer));
}


void wrtn_grammar::read_xml(xmlNode * node) {

  patterns.clear();
  pat_map.clear();

  if (node == NULL) { throw xml_parse_error("wrtn_grammar::read_XML: document is empty"); }

  if (xmlStrcmp(node->name, WRTN_GRAMMAR_ELEM)) {
    throw xml_parse_error("wrtn_grammar:read_XML: wrong document type");
  }

  string mainname;

  char * text = xmlGetProp(node, MAIN_ATTR);
  if (text) {
    mainname = text;
    xmlFree(text);
  }

  text = xmlGetProp(node, SIZE_ATTR);
  if (text) {
    patterns.reserve(lexical_cast<int>(text));
    xmlFree(text);
  }

  node = node->children;

  int syntno = 0;
  while (node) {

    if (xmlStrcmp(node->name, WRTN_PATTERN_ELEM) == 0) {

      patterns.resize(syntno + 1);
      patterns[syntno].read_xml(node, lingdef);
      pat_map[patterns[syntno].get_name()] = syntno;
      ++syntno;
    }
    node = node->next;
  }

  if (patterns.empty()) {
    throw runtime_error("wrtn_grammar::read xml: grammar is empty");
  }

  if (! mainname.empty()) {
    axiom = pat_map[mainname];
  } else { axiom = 0; }
}

