#ifndef WRTN_CHART_H
#define WRTN_CHART_H


#include <string>

#include <outilex/unique_sequence.h>
#include <outilex/sentence_fsa.h>
#include <outilex/syntref.h>

/**
 * @brief struct wrtn_match defines a match of wrtn grammar applied on a text fsa
 */

struct wrtn_match {

  wrtn_match(const std::string & n, int t, const synt_path_type & p,
            const std::vector<std::string> & o, double w_)
    : name(n), to(t), path(p), out(o), w(w_) {}
 
  /* sort by pattern name first, 
   *   then longest match first,
   *   then better weight
   *   then path and output
   * sort order is important
   */
  bool operator<(const wrtn_match & b) const { 
    if (name != b.name) { return name < b.name; }
    if (to != b.to) { return to > b.to; }
    if (w != b.w) {return w > b.w;}
    if (path != b.path) { return path < b.path; }
    return out < b.out;
  }

  void dump(std::ostream &os) const;
  
  std::string name;
  int to;
  synt_path_type path;
  std::vector<std::string> out;
  double w;
};

/**
 * @brief Class wrtn_chart defines a chart for wrtn parser
 */

struct wrtn_chart {

public:

  typedef unique_sequence<wrtn_match> node_type;

  typedef unique_sequence<wrtn_match>::iterator match_iterator;
  typedef unique_sequence<wrtn_match>::const_iterator const_match_iterator;

  typedef unique_sequence<wrtn_match>::val_iterator match_by_val_iterator;

public:

  wrtn_chart() : fsa(), tab() {}
  //wrtn_chart(const sentence_fsa & fsa_) : fsa(fsa_), tab(fsa.size()) {}

  void eat(sentence_fsa & fsa_) {
    fsa.swap(fsa_);
    tab.clear(); tab.resize(fsa.size());
  }

  int size() const { return fsa.size(); }

  unique_sequence<wrtn_match> & operator[](int q) { return tab[q]; }
  const unique_sequence<wrtn_match> & operator[](int q) const { return tab[q]; }

  int add_match(int q, const wrtn_match & match) {
    return tab[q].add(match); // -1 if already here
  }
 

  /* find first (i.e. best) match for a given syntname begininig at state q
   * all match are first sorted by syntname, then by their quality
   * best to worst (according to wrnt_match::operator<)
   * so match following the returned match iterator are of syntname also (if any)
   */ 

  match_by_val_iterator find_best_match(int q, const std::string & syntname) const;

  match_by_val_iterator by_val_end(int q) const { return tab[q].val_end(); }


  /* chart cleanup
   * * suprress all matches wich are not part of
   *     a match whose name is syntname (typically axiom of the grammar)
   * * if two match have the same pattern name and the same destination state
   *   but differs in weight, keep the best one
   *   (disregarding their path and output).
   *
   * TODO: add flags to tune the cleaning ...
   */
  void cleanup(const std::string & syntname, bool longest_match = false);

public:

  sentence_fsa fsa;
  std::vector<unique_sequence<wrtn_match> > tab;

protected:
  // internal routine used by cleanup
  void keep_it(wrtn_match & m, std::vector<node_type> & newtable);
};



#endif
