#ifndef _WPARSING_HELPER_H_
#define _WPARSING_HELPER_H_

#include <string>
#include <iostream>

#include <boost/filesystem/path.hpp>
#include <boost/scoped_ptr.hpp>

#include <outilex/text_fsa.h>
#include <outilex/wrtn_chart.h>
#include <outilex/wrtn_parser.h>



/* exemple of wparser functors
 */

/* do-nothing functor (can be useful for benchmarking parsing) */

struct wchart_null_functor {
  void operator()(const wrtn_chart & chart, int res) {}
};


/* serialize the chart to a file
 */

#include <outilex/xml_text_chart.h>

struct wchart_serializer {

  xml_ochart & ochart;

  wchart_serializer(xml_ochart & ochart_)
    : ochart(ochart_) {}

  void operator()(const wrtn_chart & chart, int res) { ochart << chart; }
};


/* chart functor which englobe another chart functor and display a progress bar
 * (on std::cout by default)
 */

#include <boost/progress.hpp>

template<typename Functor>
struct progress_displayer {

  Functor & f;
  boost::progress_display progress;
  
  progress_displayer(Functor & f_, int size, std::ostream & os = std::cout) 
    : f(f_), progress(size, os) {}

  void operator()(wrtn_chart & chart, int res) { f(chart, res); ++progress; }

  unsigned long count() const { return progress.count(); }
};

template<typename Functor>
progress_displayer<Functor> make_progress_displayer(Functor & f, int size,
                                                    std::ostream & os = std::cout) {
  return progress_displayer<Functor>(f, size, os);
}


template<typename F1, typename F2>
struct tee_functor {

  F1 & f1;
  F2 & f2;

  tee_functor(F1 & f1_, F2 & f2_) : f1(f1_), f2(f2_) {}

  void operator()(wrtn_chart & chart, int res) {
    f1(chart, res); f2(chart, res);
  }
};

template<typename F1, typename F2>
tee_functor<F1,F2> make_tee_functor(F1 & f1, F2 & f2) {
  return tee_functor<F1,F2>(f1,f2);
}


template<typename Functor>
int wrtn_parse(wrtn_parser & parser, itext_fsa & itext, Functor & f,
               bool longest_match = false,
               double maxtime = 0., bool verbose = false) {

  sentence_fsa fsa;
  wrtn_chart chart;
  const std::string & axiom = parser.gram.start_name();
  int sentenceno = 0;

  while (itext >> fsa) {
  
    if (verbose) {
      std::cerr << "#" << sentenceno << " " << fsa.text << "\n"; 
    }
    
    chart.eat(fsa);

    int res = parser.parse(chart, maxtime);
    chart.cleanup(axiom, longest_match);

    f(chart, res);
    ++sentenceno;
  }

  return sentenceno;
}


template<typename Functor>
inline int wrtn_parse(itext_fsa & itext, wrtn_grammar & gram, Functor & f,
                      int parser_flags, bool longest_match, double timeout, bool verbose) {

  wrtn_parser parser(gram, parser_flags);
  return wrtn_parse(parser, itext, f, longest_match, timeout, verbose);
}


template<typename Functor>
inline int wrtn_parse(itext_fsa & itext, wrtn_grammar & gram, Functor & f,
                      int parser_flags, bool longest_match) {

  return wrtn_parse(itext, gram, f, parser_flags, longest_match, 0., false);
}

template<typename Functor>
inline int wrtn_parse(itext_fsa & itext, wrtn_grammar & gram, Functor & f,
                      int parser_flags, bool longest_match, double maxtime) {

  return wrtn_parse(itext, gram, f, parser_flags, longest_match, maxtime, false);
}

template<typename Functor>
inline int wrtn_parse(itext_fsa & itext, wrtn_grammar & gram, Functor & f,
                      int parser_flags, bool longest_match, bool verbose) {

  if (verbose) { std::cerr << "verbose\n"; }
  return wrtn_parse(itext, gram, f, parser_flags, longest_match, 0., verbose);
}


#if 0
template<typename Functor>
void wrtn_parse(const boost::filesystem::path & txtpath,
                const boost::filesystem::path & grampath,
                Functor & f,
                ling_def * lingdef, int parser_flags = wrtn_parser::SURF_PARSING,
                bool longest_match = false, double maxtime = 0., bool verbose = false) {


  boost::scoped_ptr<itext_fsa> p_itext(new_itext_fsa(txtpath, lingdef));
  itext_fsa & itext = *p_itext;

  wrtn_grammar gram(grampath, lingdef);

  wrtn_parse(itext, gram, f, parser_flags, longest_match, maxtime);
}
#endif


#endif
