#include <iostream>
#include <vector>
#include <algorithm>
#include <iterator>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>
#include <outilex/stringtok.h>
#include <boost/lexical_cast.hpp>

using namespace std;
using namespace boost;

namespace fs = boost::filesystem;
namespace {

char * progname = NULL;

int CONTEXT_RIGHT = 80;
int CONTEXT_LEFT  = 50;


void usage(ostream & os, int ret) {
  os << "usage: " << progname << " <concordidx> [-left <left-size>][-right <right-size>][ -o <res>][-dontsort]\n";
  exit(ret);
}

void usage() { usage(cout, 0); }
void bad_args() { cerr << "\nbad arguments.\n\n"; usage(cerr, 1); }



void my_sort(istream & is, ostream & os) {

  string line;
  vector<string> text;

  while (getline(is, line)) {
    text.push_back(line);
  }

  std::sort(text.begin(), text.end());
  std::copy(text.begin(), text.end(), ostream_iterator<string>(os, "\n"));
}

void html_prolog(ostream & os) {

  os << 
    "<html>\n"
    "<head>\n"
    "  <meta http-equiv='Content-Type' content='text/html; charset=UTF-8'>\n"
    "  <title>Concordances</title>\n"
    "</head>\n"
    "<body>\n"
    "<table border='0' cellpadding='0' width='100%' style='font-family: monospace'>\n"
    "\n";
}

void html_epilog(ostream & os) {
  os <<
    "</table>\n"
    "</body>\n"
    "</html>\n";
}

void escape_html(std::string & text) {
  
  string::size_type pos = 0;

  while ((pos = text.find_first_of("<>", pos)) != string::npos) {
    
    if (text[pos] == '<') {
      text.replace(pos, 1, "&lt;");
    } else {
      text.replace(pos, 1, "&gt;");
    }
    pos = pos + 4;
  }
}

void dump_concord_line(ostream & os, const string & line, int & lineno) {

  vector<string> tab;
  stringtok(line, "\t", back_inserter(tab));

  /* 0: center, 1: right, 2: left, ... */

  if (tab.size() < 4) {
    cerr << "bad concord line : " << line << "\n";
    return;
  }

  string & center = tab[0], right = tab[1], left = tab[2];

  if (right.size() > CONTEXT_RIGHT) { right.erase(CONTEXT_RIGHT); }
  if (left.size() > CONTEXT_LEFT) { left.erase(0, left.size() - CONTEXT_LEFT); }


  escape_html(left); escape_html(center); escape_html(right);

  os << "<tr>"
    << "<td>(" << lineno++ << ")</td>"
    << "<td align='right' nowrap>" << left << "</td>"
    << "<td nowrap>&nbsp;<a href='" << tab[3] << "'>" << center << "</a>&nbsp;"
    << right << "</td></tr>\n";
}


void make_html_concordances(istream & is, ostream & os) {
  html_prolog(os);
  string line;
  int lineno = 1;
  while (getline(is, line)) {
    dump_concord_line(os, line, lineno);
  }
  html_epilog(os);
}

} // namespace ""



int main(int argc, char ** argv) try {

  fs::path idxpath, respath;
  bool sort_idx = true;
  
  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {

    if (**argv == '-') {

      if (strcmp(*argv, "-h") == 0) {

        usage();

      } else if (strcmp(*argv, "-o") == 0) {

        argv++,argc--;
        if (! argc) { bad_args(); }
        respath = *argv;

      } else if (strcmp(*argv, "-dontsort") == 0) {

        sort_idx = false;

      } else if(strcmp(*argv, "-left") == 0){
        argv++; argc--;   
        if(!argc){bad_args();}
        CONTEXT_LEFT = lexical_cast<int>(*argv);
      }else if(strcmp(*argv, "-right") == 0){
        argv++; argc--;
        if(!argc){bad_args();}
        CONTEXT_RIGHT = lexical_cast<int>(*argv);
      }
      else { bad_args(); }
    }
    else{
      idxpath = *argv;
    }

    argv++, argc--;
  }

  if (idxpath.empty()) { bad_args(); }

  if (respath.empty()) { respath = idxpath.branch_path() / "concord.html"; }


  if (sort_idx) {
    
    cout << "preliminary sort ...\n";
    fs::path sortedpath = idxpath.branch_path() / (idxpath.leaf() + ".sort");
    
    fs::ifstream is(idxpath);
    fs::ofstream os(sortedpath);

    my_sort(is, os);

    idxpath = sortedpath;

    cout << "done.\n";
  }

  cout << "making concordances ...\n";
  fs::ifstream in(idxpath);
  fs::ofstream out(respath);

  make_html_concordances(in, out);
  cout << "done.\nresult in " << respath.string() << ".\n";

} catch (std::exception & e) {
  cerr << "fatal error : " << e.what() << endl;
  exit(1);
}

