#include <iostream>
#include <fstream>
#include <stdexcept>
#include <sstream>
#include <vector>

#include <boost/progress.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>

#include <outilex/text_fsa.h>
#include <outilex/bin_text_fsa.h>
#include <outilex/sentence_fsa.h>
#include <outilex/wrtn_grammar.h>
#include <outilex/wparsing_helper.h>
#include <outilex/fsa_decoration.h>


using namespace std;
using namespace boost;
namespace fs = boost::filesystem;


namespace {

char * progname;


void usage(ostream & os, int ret) {
  os << "usage: " << progname
    << 
    " -l <lingdef> -gram <fst> [-v][-longest-match][-r][-ipath][-iout][-o <outputres>] <txtfsa>\n"
    "\n"
    "with:\n"
    " <txtfsa>           : the input text fsa\n"
    " -lingdef <lingdef> : specify the lingdef tagset description file\n"
    " -v                 : verbose mode (primary for debugging purpose)\n"
    " -r                 : replace mode, replace the matching pathes by newly created transitions\n"
    " -longest-match     : proceed only with longest matches if there is more than one match\n"
    "                      begining in the same state in the input text fsa\n"
    " -o <otextfsa>      : specify the name of the resulting textfsa (default to <txtfsa>.deco\n"
    "\n"
    "unuseful options:\n"
    " -iout              : if there is plural (plusieurs in French) matches with the same path\n"
    "                      in the input fsa\n"
    "                      but differing in their output, proceed with only one of thoses\n"
    " -ipath             : if there is plural matches on the same segment (i.e. same init\n"
    "                      and destination state) in the input fsa but differing in their pathes\n"
    "                      (i.e. in their transitions), proceed with only one of thoses\n"
    "\n"
    "apply a decoration grammar to a text fsa and create a new text fsa decorated with new\n"
    "lexical transitions as specified by the grammar.\n\n";

  exit(ret);
}

void usage() { usage(cout, 0); }
void bad_args() {
  cerr << "bad arguments.\n\n";
  usage(cerr, 1);
}

}; // namespace ""



int main(int argc, char ** argv) try {

  fs::path txtpath, lingdefpath, wrtnpath, opath;
  int PARSER_FLAGS = wrtn_parser::SURF_PARSING; //wrtn_parser::IGNORE_DIFF_PATHS;
  bool longest_match = false;
  int deco_flags = 0;
  bool verbose = false;
 
  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }

  progname = *argv;


  argv++, argc--;

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {
      
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-o") {
 
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      opath = fs::path(*argv, fs::native);
    
    } else if ((arg == "-gram") || (arg == "-rtn")) {
      
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      wrtnpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-longest-match") {
    
      longest_match = true;

    } else if (arg == "-ipath") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_PATHS;

    } else if (arg == "-iout") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_OUTPUTS;

    } else if (arg == "-h") {
    
      usage();
    
    } else if (arg == "-r") {
    
      deco_flags |= deco_replace;
 
    } else if (arg == "-v") {
    
      verbose = true;
      deco_flags |= deco_verbose;
    
    } else {

      txtpath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (txtpath.empty() || lingdefpath.empty() || wrtnpath.empty()) { bad_args(); }

  if (opath.empty()) {
    string resname = txtpath.leaf() + ".deco";
    opath = txtpath.branch_path() / resname;
  }
  
  
  ling_def lingdef(lingdefpath);

  scoped_ptr<itext_fsa> p_itext(new_itext_fsa(txtpath, & lingdef));
  itext_fsa & itext = *p_itext;

  wrtn_grammar gram(wrtnpath, & lingdef);

  int total = itext.size();
  bin_otext_fsa otext(opath, total);

  if (deco_flags & deco_verbose) { cerr << "verbose mode (main)\n"; }

  fsa_decorizer decorizer(otext, gram.start_name(), deco_flags);
  progress_displayer<fsa_decorizer> displayer(decorizer, total);

  boost::timer tmr;

  int nbsentence = wrtn_parse(itext, gram, displayer, PARSER_FLAGS, longest_match, verbose);


  cout << "done. " << nbsentence << " sentences.\n"
     << decorizer.nbmatch << " new transitions. " << tmr.elapsed() << "s.\n"
     << "resulting text fsa in " << opath.string() << "\n";

  return 0;

} catch (exception & e) {

  cerr << "fatal error :" << e.what() << endl; exit(1);

} catch (...) { cerr << "caught an OVNI?\n"; exit(1); }

