#include <iostream>
#include <string>
#include <vector>

#include <outilex/sentence_fsa.h>
#include <outilex/wrtn_chart.h>
#include <outilex/concordances.h>

using namespace std;

namespace {

struct same_dest {

  int to;

  same_dest(int dest) : to(dest) {}

  template<typename Trans>
  bool operator()(const Trans & tr) const { return tr.to() == to; }
};

struct same_pos {

  pos_def * pos;

  same_pos(pos_def * p) : pos(p) {}

  template<typename Trans>
  bool operator()(const Trans & tr) const { return tr.in().pos == pos; }
};

struct compare_dest {
  template<typename Trans>
  bool operator()(const Trans & a, const Trans & b) const { return a.to() < b.to(); }
};

sentence_fsa::const_trans_iterator find_best_trans(const sentence_fsa & fsa, int q) {

  const sentence_fsa::const_trans_iterator begin = fsa.trans_begin(q), end = fsa.trans_end(q);

  /* first, try to find a lex transition */
  pos_def * lexpos = fsa.lingdef->lex_pos();

  sentence_fsa::const_trans_iterator tr = find_if(begin, end, same_pos(lexpos));
  
  if (tr != end) { // found
    return tr;
  }
  
  /* else, return the shortest one */

  return min_element(begin, end, compare_dest());
}

sentence_fsa::const_trans_iterator find_lex_trans(const sentence_fsa & fsa, int from, int to) {
  return find_if(fsa.trans_begin(from), fsa.trans_end(from), same_dest(to));
}


/* on stock (sous forme de liste chainee) les contextes gauche et droit pour chaque etat
 * une fois pour pas a avoir a les recalculer pour chaque concordance
 * 
 */


struct context {
  const string * text;
  
  /* etat suivant ou precedent suivant qu'on est dans un ctxt gauche ou droit
   * -1 si erreur */
  int q;

  context() : text(NULL), q(-1) {}

  const string & str() const { return *text; }
};


void precompute_contexts(const sentence_fsa & fsa,
                         vector<context> & left, vector<context> & right) {

  int size = fsa.size();

  left.clear(); right.clear();
  left.resize(size);
  right.resize(size);


  int q = 0;
  while (! fsa.final(q)) {

    sentence_fsa::const_trans_iterator tr = find_best_trans(fsa, q);
    if (tr == fsa.trans_end(q)) {
      cerr << "error : in precompute context : txt fsa is not pruned\n";
      break;
    }

    left[tr->to()].q = q;
    left[tr->to()].text = & (tr->in().form);
  
    right[q].q = tr->to();
    right[q].text = & (tr->in().form);

    q = tr->to();
  }

  /* look for unvisited states */
  for (q = 1; q < size - 1; ++q) {
    
    if (right[q].q == -1) { // pas de contexte droit

      sentence_fsa::const_trans_iterator tr = find_best_trans(fsa, q);

      if (tr != fsa.trans_end(q)) {
        right[q].q = tr->to();
        right[q].text = & tr->in().form;
        break;
      }
    }
 
    if (left[q].q == -1) { // pas de context gauche
      
      for (int from = q - 1; from >= 0; --from) {

        sentence_fsa::const_trans_iterator tr = find_lex_trans(fsa, from, q);

        if (tr != fsa.trans_end(from)) {
          left[q].q = from;
          left[q].text = & tr->in().form;
        }
      }
    }
  }
}


/* easy one */
void dump_right(const vector<context> & right, int q, ostream & os) {

  if (right[q].q == -1) { os << ' '; return; }

  while (right[q].q != -1) {
    os << right[q].str() << ' ';
    q = right[q].q;
  }
}


/* harder one */
void dump_left(const vector<context> & left, int q, ostream & os) {
  
  if (left[q].q == -1) { os << ' '; return; }

  vector<int> stack;
  while (left[q].q != -1) {
    stack.push_back(q);
    q = left[q].q;
  }
  
  while (! stack.empty()) {
    os << left[stack.back()].str() << ' ';
    stack.pop_back();
  }
}


inline void dump_trans(const sentence_fsa::const_transition & tr, ostream & os, int flags) {
  if (flags & CONCORD_SHOW_TAGS) {
    os << tr.in() << ' ';
  } else {
    os << tr.in().form << ' ';
  }
}

void dump_match(const wrtn_chart & chart, const wrtn_match & match, ostream & os, int flags) {

  if (flags & CONCORD_SHOW_TREE) { os << '(' << match.name << ' '; }

  int size = match.path.size();
  for (int i = 0; i < size; ++i) {
 
    if (flags & CONCORD_SHOW_OUTPUTS && ! match.out[i].empty()) { os << match.out[i] << ' '; }

    const syntref & ref = match.path[i];
 
    if (ref.transno < 0) {
      dump_match(chart, chart[ref.qno][- ref.transno - 1], os, flags);
    } else {
      dump_trans(chart.fsa.get_trans(ref.qno, ref.transno), os, flags);
    }
  }

  if (flags & CONCORD_SHOW_OUTPUTS && ! match.out[size].empty()) { os << match.out[size] << ' '; }

  if (flags & CONCORD_SHOW_TREE) { 
    os << match.name << ')'; 
    if (flags & CONCORD_SHOW_WEIGHTS) { os << '/' << match.w; }
    os << ' ';
  }
}

} // namespace ""



/* concord index format is : center \t right \t left \t fromid \t toid \n
 */

int wchart_make_concordances_index(const wrtn_chart & chart, ostream & os,
                                   const string & axiom, int flags) {

  //cerr << "make_concord_idx\n";

  const sentence_fsa & fsa = chart.fsa;

  int nbmatch = 0;

  wrtn_match firstmatch(axiom,
                        numeric_limits<int>::max(),
                        synt_path_type(),
                        vector<string>(),
                        numeric_limits<double>::max());

  int size = fsa.size();

  vector<context> left, right;
  precompute_contexts(fsa, left, right);

  // ignore inner matches
  int maxto = 0, minfrom = size;
  for (int q = 0; q < size; ++q) {

    // cerr << "q = " << q << endl;


    //if (keep_inner_matches) { maxto = 0; }

    wrtn_chart::match_by_val_iterator it = chart[q].lower_bound(firstmatch),
      end = chart[q].val_end();


    if (it != end) {
      if (maxto < it->to) {
        minfrom = q;
        maxto = it->to;
      }
    }

    ostringstream oss;
    string old;

    while (it != end && it->name == axiom) {
 
      /* ignore inner matches */
      if (flags & CONCORD_LONGEST_MATCH 
          && ((it->to < maxto) || (q > minfrom))) { break; }

      oss.str("");

      // dump match
      dump_match(chart, *it, oss, flags);
      oss << '\t';

      dump_right(right, it->to, oss);
      oss << '\t';

      dump_left(left, q, oss);
      oss << '\t';

      oss << fsa.pos(q) << ' ' << fsa.pos(q);

      string nouvo(oss.str());
      
      if (nouvo != old) {
        os << nouvo << '\n';
        old = nouvo;
        ++nbmatch;
      }

      ++it;
    }
  }

  //cerr << "out of make_concord_idx\n";
  return nbmatch;
}


