#include <iostream>
#include <fstream>
#include <stdexcept>
#include <sstream>
#include <vector>

#include <boost/progress.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/fstream.hpp>

#include <outilex/text_fsa.h>
#include <outilex/sentence_fsa.h>
#include <outilex/wrtn_grammar.h>
#include <outilex/wrtn_chart.h>
#include <outilex/wrtn_parser.h>
#include <outilex/xml_text_chart.h>


namespace {

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

char * progname;


void usage() {
  cerr << "usage: " << progname
    << " -l <lingdef> -gram <fst> [-lgst][-ipath][-iout][-dontsurf][-o <outputres>] <txtfsa>\n";
  exit(1);
}


}; // namespace ""

int main(int argc, char ** argv) try {

  fs::path txtpath, lingdefpath, wrtnpath, opath;
  bool longest_match = false;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  int PARSER_FLAGS = wrtn_parser::SURF_PARSING;

  progname = *argv;

  argv++, argc--;

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-o") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      opath = fs::path(*argv, fs::native);
    
    } else if (arg == "-gram") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      wrtnpath = fs::path(*argv, fs::native);
 
    } else if (arg == "-ipath") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_PATHS;

    } else if (arg == "-iout") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_OUTPUTS;

    } else if (arg == "-dontsurf") {

      PARSER_FLAGS &= ~(wrtn_parser::SURF_PARSING);

    } else if (arg == "-lgst") {
    
      longest_match = true;

    } else if (arg == "-h") {
    
      usage();
    
    } else {

      txtpath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (txtpath.empty() || lingdefpath.empty() || wrtnpath.empty()) { usage(); }

  if (opath.empty()) { opath = fs::change_extension(txtpath, ".chart"); }


  ling_def lingdef(lingdefpath);

  scoped_ptr<itext_fsa> p_itext(new_itext_fsa(txtpath, & lingdef));
  itext_fsa & itext = *p_itext;

  int total = itext.size();

  xml_ochart ochart(opath, total);


  wrtn_grammar wrtn(wrtnpath, & lingdef);
  string axiom = wrtn.start_name();

  wrtn_parser parser(wrtn, PARSER_FLAGS);

  sentence_fsa fsa;

  int sentenceno = 0;
  //int totalmatches = 0;
  progress_display show_progress(total, cout);
  boost::timer tmr;

  while (itext >> fsa) {

    //cerr << "sentence #" << sentenceno <<  "/" << total << endl;

    wrtn_chart chart;
    chart.eat(fsa);
   
    if (! chart.fsa.empty()) {
      parser.parse(chart);
      chart.cleanup(axiom, longest_match);
    }

    ochart << chart;

    ++sentenceno;
    ++show_progress;
  }

  cout << "done. " << sentenceno << "/" << total << " sentences parsed. "
    << tmr.elapsed() << "s.\n";
 
   // << "found " << totalmatches << " matching sequences, "

  cout << "resulting chart stored in " << opath.string() << endl;

  return 0;

} catch (exception & e) {

  cerr << "exception caught :" << e.what() << endl; exit(1);

} catch (...) { cerr << "caught an OVNI?\n"; exit(1); }

