#include <boost/lexical_cast.hpp>
#include <outilex/xml_text_uchart.h>
#include <outilex/xml-names.h>
#include <outilex/xml_format_uchart.h>
#include <outilex/uchart.h>


using namespace std;
using namespace boost;
using namespace xmlnames;

namespace fs = boost::filesystem;

int LINK_WITH_XML_TEXT_UCHART;


namespace {

bool xml_guess(const boost::filesystem::path & path) try {

  xmlreader reader(path);

  while (reader.node_type() != XML_READER_TYPE_ELEMENT) { reader.read(); reader.check(); }

  if (xmlStrcmp(reader.const_name(), TEXT_UCHART_ELEM) != 0) { return false; }

  cerr << "xml format detected\n";
  return true;

} catch (exception & e) {
  return false;
}

itext_uchart * new_xml_itext_uchart(const fs::path & path, ling_def * ldef) {
  return new xml_itext_uchart(path, ldef);
}
text_chart_reader_registerer ixml_registration(xml_guess, new_xml_itext_uchart);

} // namespace ""


void xml_itext_uchart::open(const fs::path & path_) {

  close();
  path = path_;
  reader.open(path);

  while (reader.node_type() != XML_READER_TYPE_ELEMENT) { reader.read(); reader.check(); }


  if (xmlStrcmp(reader.const_name(), TEXT_UCHART_ELEM) != 0) {
    throw xml_parse_error("xml_itext_uchart::open: " + path.string() + ": bad document type");
  }

  char * text = reader.get_attribute(SIZE_ATTR);
  if (text) {
    size_ = lexical_cast<int>(text);
    xmlFree(text);
  } else { size_ = -1; }

  //cerr << "xmlchartopen, size = " << size_ << endl;
  pos = 0;
}

void xml_itext_uchart::close() {
  path = fs::path();
  reader.close();
  size_ = pos = -1;
}


bool xml_itext_uchart::read_next(uchart & chart) {

  if (! reader || pos == -1) {
    return false;
  }

  int ret = 1;

  do {
  
    if (reader.node_type() == XML_READER_TYPE_ELEMENT
        && xmlStrcmp(reader.const_name(), UCHART_ELEM) == 0) {
 
      xmlNode * node = reader.expand();
      uchart_read_xml(node, chart, lingdef);

      ret = reader.next(); // skip subtree
      reader.check();

      pos++;
      return true;

    } else { ret = reader.read(); }
  
  } while (ret == 1);


  reader.check();

  /* EOF */
  pos = -1;
  return false;
}

void xml_itext_uchart::rewind() { open(path); }


void xml_itext_uchart::seek(int offset) {

  if (! reader) {
    throw runtime_error("xml_text_uchart::seek: no stream");
  }
  if (offset < pos) { rewind(); }

  int ret = 1;
  while (ret == 1 && pos < offset) {

    if (reader.node_type() == XML_READER_TYPE_ELEMENT
        and xmlStrcmp(reader.const_name(), UCHART_ELEM) == 0) {
 
      ret = reader.next();
      ++pos;
 
    } else { ret = reader.read(); }

    reader.check();
  }

  if (ret != 1) {
    pos = -1;
    throw runtime_error("xml text uchart::seek: EOF");
  }
}




void xml_otext_uchart::open(std::ostream & os, int size) {

  close();
 
  writer.open(os);
  writer.set_indent(1);

  writer.start_document();
  writer.start_element(TEXT_UCHART_ELEM);

  if (size != -1) {
    writer.write_attribute(SIZE_ATTR, lexical_cast<string>(size));
  }
}


void xml_otext_uchart::open(const fs::path & path, int size, int compression) {

  close();
  
  writer.open(path, compression);
  writer.set_indent(1);

  writer.start_document();
  writer.start_element(TEXT_UCHART_ELEM);

  if (size != -1) {
    writer.write_attribute(SIZE_ATTR, lexical_cast<string>(size));
  }
}


void xml_otext_uchart::close() {
  if (writer) {
    writer.end_element(); // text-uchart
    writer.end_document();
    writer.close();
  }
}


void xml_otext_uchart::write(const uchart & chart) {
  uchart_write_xml(writer, chart);
}

