#include <boost/lexical_cast.hpp>

#include <outilex/xml.h>
#include <outilex/xml-names.h>
#include <outilex/usyntagm_pattern.h>
#include <outilex/generic_fst.h>

using namespace std;
using namespace boost;
using namespace xmlnames;

namespace fs = boost::filesystem;


usyntagm_pattern::usyntagm_pattern(const generic_fst & fst, ling_def * ldef) {

  //set_name(fst.get_name());
  set_name(fst.name);
  lingdef = ldef;

  states.clear();
  states.resize(fst.size());

  for (int i = 0; i < fst.size(); ++i) {

    final(i) = fst.final(i);

    if (final(i)) {
      const generic_fst::outputs_type & outputs = fst.final_outputs(i);
      for (generic_fst::outputs_type::const_iterator it = outputs.begin();
           it != outputs.end(); ++it) {
        final_outputs(i).insert(output_type(*it));
      }
    }

    for (generic_fst::transitions::const_iterator tr = fst.trans_begin(i);
         tr != fst.trans_end(i); ++tr) {
      add_trans(i, input_type(tr->in(), lingdef), output_type(tr->out()), tr->to());
    }
  }
}


void usyntagm_pattern::read(const fs::path & path) {

  string fname = path.native_file_string();

  xmlDocPtr doc = xmlParseFile(fname.c_str());

  if (doc == NULL) {
    throw xml_parse_error("usynt-pattern::read_XML: " + fname + " not successfully parsed");
  }

  xmlNodePtr node = xmlDocGetRootElement(doc);

  if (xmlStrcmp(node->name, USYNTAGM_PATTERN_ELEM)) {
    throw xml_parse_error("usynt-pattern::read_XML: " + fname + " wrong document type");
  }

  try {
    read_xml(node, lingdef);
  } catch (exception & e) {
    xmlFreeDoc(doc);
    throw;
  }
  xmlFreeDoc(doc);
}


void usyntagm_pattern::write(const fs::path & path) const {

  xmlwriter writer(path);
  writer.set_indent(true);

  writer.start_document();
  write_xml(writer);
  writer.end_document();
}



void usyntagm_pattern::transition::write_xml(xmlwriter & writer) const {

  writer.start_element(TRANS_ELEM);
  writer.write_attribute(TO_ATTR, lexical_cast<string>(to_));
 
  writer.start_element(IN_ELEM);
  in_.write_xml(writer);
  writer.end_element();

  writer.start_element(OUT_ELEM);
  out_.write_XML(writer);
  writer.end_element();

  writer.end_element();
}


void usyntagm_pattern::transition::read_xml(xmlNodePtr node, ling_def * lingdef) {

  //cerr << "upattern::read_trans\n";

  char * text = xmlGetProp(node, TO_ATTR);
  to_ = lexical_cast<int>(text);
  xmlFree(text);

  bool okin = false, okout = false;

  node = node->children;
  while (node) {

    if (xmlStrcmp(node->name, IN_ELEM) == 0) {

      xmlNode * cur = node->children;

      while (cur) {
        if (xmlStrcmp(cur->name, SUBCALL_ELEM) == 0
            || xmlStrcmp(cur->name, EPSILON_ELEM) == 0
            || xmlStrcmp(cur->name, LEXMASK_ELEM) == 0) {
          //dd    cerr << "name = " << cur->name << endl;
          in_.read_xml(cur, lingdef);
          okin = true;
        }
        cur = cur->next;
      }

    } else if (xmlStrcmp(node->name, OUT_ELEM) == 0) {

      //cerr << "out\n";

      xmlNode * cur = node->xmlChildrenNode;

      while (cur) {
        if (xmlStrcmp(cur->name, output_type::xml_name()) == 0) {
          //cerr<< "name = " << cur->name << endl;
          out_.read_XML(cur);
          okout = true;
        }
        cur = cur->next;
      }
    }

    node = node->next;
  }

  //cerr << "okin = " << okin << " okout = " << okout << endl;

  if (!okin || !okout) {
    throw xml_parse_error("syntagm_pattern::read_XML: bad transition");
  }
}


void usyntagm_pattern::state::write_final_outputs(xmlwriter & writer) const {

  writer.start_element(FINAL_OUTPUTS_ELEM);
  for (outputs_type::const_iterator it = final_outputs.begin(); it != final_outputs.end(); ++it) {
    it->write_XML(writer);
  }
  writer.end_element();
}


void usyntagm_pattern::state::read_final_outputs(xmlNodePtr node) {
  node = node->xmlChildrenNode;
  while (node) {
    if (xmlStrcmp(node->name, output_type::xml_name()) == 0) {
      output_type out(node);
      final_outputs.insert(out);
    }
    node = node->next;
  }
}



void usyntagm_pattern::state::write_XML(xmlwriter & writer, int id) const {

  writer.start_element(STATE_ELEM);

  if (id != -1) {
    writer.write_attribute(ID_ATTR, lexical_cast<string>(id));
  }

  if (final) {
    writer.write_attribute(FINAL_ATTR, "1");
  }
  writer.write_attribute(SIZE_ATTR, lexical_cast<string>(trans.size()));

  if (! final_outputs.empty()) { write_final_outputs(writer); }

  for_each(trans.begin(), trans.end(), XML_Writer(writer));

  writer.end_element();
}



void usyntagm_pattern::state::read_XML(xmlNodePtr node, ling_def * lingdef) {

  clear();

  char * text = xmlGetProp(node, FINAL_ATTR);

  if (text && text[0] == '1') {
    final = true;
  } else { final = false; }

  if (text) { xmlFree(text); }

  text = xmlGetProp(node, SIZE_ATTR);
  if (text) {
    trans.reserve(lexical_cast<int>(text));
    xmlFree(text);
  }
  
  node = node->children;

  while (node) {

    if (xmlStrcmp(node->name, TRANS_ELEM) == 0) {

      int size = trans.size();

      try {

        trans.resize(size + 1);
        trans[size].read_xml(node, lingdef);

      } catch (xml_parse_error & e) {

        cerr << "unable to load transition: " << e.what() << '\n';
        trans.resize(size);
      }

    } else if (xmlStrcmp(node->name, FINAL_OUTPUTS_ELEM) == 0) {
      read_final_outputs(node);
    }
    node = node->next;
  }
}


void usyntagm_pattern::write_xml(xmlwriter & writer) const {

  writer.start_element(USYNTAGM_PATTERN_ELEM);
  writer.write_attribute(NAME_ATTR, name);
  writer.write_attribute(SIZE_ATTR, lexical_cast<string>(size()));

  for (int q = 0; q < states.size(); ++q) { states[q].write_XML(writer, q); }
  writer.end_element();
}


void usyntagm_pattern::read_xml(xmlNodePtr node, ling_def * ldef) {
  
  lingdef = ldef;
  clear();

  char * text = xmlGetProp(node, NAME_ATTR);
  name = text;
  xmlFree(text);

  text = xmlGetProp(node, SIZE_ATTR);
  if (text) {
    states.reserve(lexical_cast<int>(text));
    xmlFree(text);
  }


  node = node->xmlChildrenNode;

  int qno = 0;
  while (node) {
    if (xmlStrcmp(node->name, STATE_ELEM) == 0) {
      states.resize(qno + 1);
      states[qno].read_XML(node, lingdef);
      qno++;
    }
    node = node->next;
  }

  //cerr << "pattern " << name << " loaded (" << size() << " states)\n";
}

