#ifndef _USYNTAGM_PATTERN_H_
#define _USYNTAGM_PATTERN_H_

#include <set>

#include <boost/filesystem/path.hpp>

#include <outilex/synt_input_type.h>
#include <outilex/weighted_constraints.h>
#include <outilex/xml.h>


class generic_fst;

class usyntagm_pattern {
public:

  typedef synt_input_type      input_type;
  typedef weighted_constraints output_type;

  typedef std::set<output_type> outputs_type;

  struct transition {

    int to_;
    input_type  in_;
    output_type out_;

    inline transition() {}
    inline transition(const input_type & in, const output_type & out, int to)
      : to_(to), in_(in), out_(out) {}

    inline transition(int to, const input_type & in, const output_type & out)
      : to_(to), in_(in), out_(out) {}

    inline int to() const { return to_; }

    inline input_type & in() { return in_; }
    inline const input_type & in() const { return in_; }

    inline output_type & out() { return out_; }
    inline const output_type & out() const { return out_; }

    void write_xml(xmlwriter & writer) const;
    void read_xml(xmlNodePtr node, ling_def * lingdef);
  };

  typedef std::vector<transition> transitions;
  typedef transitions::iterator trans_iterator;
  typedef transitions::const_iterator const_trans_iterator;

  struct state {

    bool final;
    outputs_type final_outputs;
    transitions trans;
  
    state() : final(false), final_outputs(), trans() {}

    inline void clear() { final = false; final_outputs.clear(); trans.clear(); }


    void write_XML(xmlwriter & writer, int id = -1) const;
    void read_XML(xmlNodePtr node, ling_def * ldef);

  protected:
    void write_final_outputs(xmlwriter & writer) const;
    void read_final_outputs(xmlNodePtr node);
  };


  inline usyntagm_pattern(ling_def * ldef = NULL) : name(), lingdef(ldef), states() {}

  usyntagm_pattern(const boost::filesystem::path & path, ling_def * ldef)
    : name(), lingdef(ldef), states() { read(path); }

  usyntagm_pattern(const generic_fst & fst, ling_def * ldef);


  inline void set_name(const std::string & n) { name = n; }
  inline const std::string & get_name() const { return name; }

  inline int start() const { return 0; }

  bool empty() const { return states.empty(); }
  inline int size() const { return states.size(); }

  inline void clear() { states.clear(); }

  inline trans_iterator trans_begin(int q) { return states[q].trans.begin(); }
  inline const_trans_iterator trans_begin(int q) const { return states[q].trans.begin(); }

  inline trans_iterator trans_end(int q) { return states[q].trans.end(); }
  inline const_trans_iterator trans_end(int q) const { return states[q].trans.end(); }

  inline void add_trans(int from, const input_type & in, const output_type & out, int to) {
    states[from].trans.push_back(transition(in, out, to));
  }

  inline bool final(int q) const { return states[q].final; }
  inline bool & final(int q) { return states[q].final; }

  inline outputs_type & final_outputs(int q) { return states[q].final_outputs; }
  inline const outputs_type & final_outputs(int q) const { return states[q].final_outputs; }

  int add_state() {
    int res = states.size();
    states.resize(res + 1);
    return res;
  }

  static inline const char * xml_name() { return "usyntagm_pattern"; }

  void write(const boost::filesystem::path & path) const;
  void write_xml(xmlwriter & writer) const;

  void read(const boost::filesystem::path & path);
  inline void read(const boost::filesystem::path & path, ling_def * ldef) {
    lingdef = ldef; read(path);
  }
  void read_xml(xmlNodePtr node, ling_def * ldef);

public:

  std::string name;
  ling_def * lingdef;

  std::vector<state> states;
};


#endif

