#ifndef _UCHART_H_
#define _UCHART_H_

#include <utility>

#include <outilex/syntref.h>
#include <outilex/featstruct.h>
#include <outilex/sentence_fsa.h>

class lingdef;

struct syntagm {

  syntagm() : name(), to(-1), fs(), path(), w(0) {}

  /* deep copy of the feature structure
   */
  syntagm(const std::string & nam, int to_, const featstruct & fs_,
          const std::vector<syntref> & path_, double w_)

    : name(nam), to(to_), fs(fs_), path(path_), w(w_) {
      fs.compress(true);
    }

  /* no deep copy of the passed featstruct is performed
   */
  syntagm(const std::string & nam, int to_, featstruct & fs_,
          const std::vector<syntref> & path_, double w_)
    : name(nam), to(to_), fs(fs_), path(path_), w(w_) {
      fs.compress(true);
    }


  //void read_XML(xmlNodePtr node, ling_def * ldef);
  //void write_XML(xmlwriter & writer) const;


  std::string name;
  int to;
  featstruct fs;
  synt_path_type path; // vector<syntref>
  double w;
};


/* chart cleanup flags */

#define CLEAN_WELLFORMED         (1)
#define CLEAN_LONGESTMATCH  (1 << 1)
#define CLEAN_BESTMATCH     (1 << 2)

class uchart {

public:
  typedef std::multimap<std::string, int> by_name_map;

  typedef by_name_map::iterator by_name_iterator;
  typedef by_name_map::const_iterator const_by_name_iterator;

  typedef std::vector<syntagm>::iterator synt_iterator;
  typedef std::vector<syntagm>::const_iterator const_synt_iterator;

  struct node_type {

    node_type() : syntagms(), synt_by_name() {}


    inline syntagm & operator[](int no) { return syntagms[no]; }
    inline const syntagm & operator[](int no) const { return syntagms[no]; }

    inline const int size() const { return syntagms.size(); }

    void find(const std::string & syntname, by_name_iterator & begin, by_name_iterator & end);
    void find(const std::string & syntname,
              const_by_name_iterator & begin, const_by_name_iterator & end) const;

    int add_match(const syntagm & synt);

    //void read_XML(xmlNode * node, ling_def * ldef);
    //void write_XML(xmlwriter & writer, int id) const;

    std::vector<syntagm> syntagms;
    by_name_map synt_by_name;
  };


  uchart() : fsa(), synt_table() {}

  uchart(const sentence_fsa & fsa);

  /* init the chart with the corresponding fsa */
  void init(const sentence_fsa & fsa);
  
  /* like init but the passed fsa is no more usable (avoid costly copy)
   */
  void eat(sentence_fsa & fsa);

  /* clear the chart (synt_table) */
  void clear();

  /* cleanup the chart :
   * suppress all the syntagms which are not part of a bigger syntagm
   * whose CAT is syntname (i.e. if syntname represents the axiom of the grammar
   * suppress all syntagm which have been computed but doesn't lead to a successfull parse)
   * * if WELLFORMED is set, keep only syntagms 'syntname' whose feature structure is well-formed
   *   (off by defaut)
   * * if longest_match is set, if several syntagms 'syntname' are present
   *   begining from the same state, keep only those which cover the bigger segment in the text 
   *   (done after well-formedness cleanup) (on by default)
   * * if byweight is set, keep the best syntagm begining from each state 
   *   (after longestmatch cleanup) (on by default)
   */
  
  void cleanup(const std::string & syntname, int flags = CLEAN_LONGESTMATCH | CLEAN_BESTMATCH);
  
  //void cleanup(const std::string & syntname, bool longest_match = true, bool byweight = true);

  /* return the chart's size 
   * (i.e. the number of states in the sentence fsa)
   */
  inline int size() const { return fsa.size(); }


  inline int add_match(int q, const syntagm & synt) { return synt_table[q].add_match(synt); }


  synt_iterator synt_begin(int q) { return synt_table[q].syntagms.begin(); }
  synt_iterator synt_end(int q)   { return synt_table[q].syntagms.end(); }

  const_synt_iterator synt_begin(int q) const { return synt_table[q].syntagms.begin(); }
  const_synt_iterator synt_end(int q)   const { return synt_table[q].syntagms.end(); }

  by_name_iterator find(int q, const std::string & name) {
    return synt_table[q].synt_by_name.find(name);
  }
  const_by_name_iterator find(int q, const std::string & name) const {
    return synt_table[q].synt_by_name.find(name);
  }

  by_name_iterator synt_by_name_end(int q) { return synt_table[q].synt_by_name.end(); }
  const_by_name_iterator synt_by_name_end(int q) const { return synt_table[q].synt_by_name.end(); }


  void find(int q, const std::string & syntname,
            const_by_name_iterator & begin, const_by_name_iterator & end) const {
    synt_table[q].find(syntname, begin, end);
  }

  void find(int q, const std::string & syntname, by_name_iterator & begin, by_name_iterator & end) {
    synt_table[q].find(syntname, begin, end);
  }


  inline syntagm & get_synt(int q, int idx) {
    if (idx < 0) { idx = -idx - 1; }
    return synt_table[q].syntagms[idx];
  }

  inline const syntagm & get_synt(int q, int idx) const {
    if (idx < 0) { idx = -idx - 1; }
    return synt_table[q].syntagms[idx];
  }

  //sentence_fsa::transition & get_lex(int q, int idx) { return fsa.get_trans(q, idx); }
  sentence_fsa::const_transition get_lex(int q, int idx) const { return fsa.get_trans(q, idx); }

  inline node_type & operator[](int q) { return synt_table[q]; }
  inline const node_type & operator[](int q) const { return synt_table[q]; }

  //static inline char * xml_name() { return "uchart"; }
  //void read_XML(xmlNode * node, ling_def * ldef);
  //void write_XML(xmlwriter & writer) const;

protected:

  //void read_synt_table_XML(xmlNode * node, ling_def * ldef);

  void keep_it(int q, int syntno, std::vector<int> * newids, std::vector<node_type> & newtable);

public:
  sentence_fsa fsa;
  std::vector<node_type> synt_table;
};

#endif
