#include <iostream>
#include <string>
#include <vector>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/lingdef.h>
#include <outilex/unitex_grf.h>
#include <outilex/generic_fst.h>
#include <outilex/ugrammar.h>

#include <outilex/flatten.h>
#include <outilex/fsa-prune.h>
#include <outilex/epsilon_removal.h>
#include <outilex/fsa-determinize.h>
#include <outilex/fsa-minimize.h>


#include <outilex/usage.h>

using namespace std;
using namespace boost;

namespace fs = boost::filesystem;

const char * USAGE_DESCRIPTION =
"usage: make-ugrammar <axiom>\n"
"\n"
"create an unification grammar from an outilex graph.\n";


namespace {


bool is_epsilon(const generic_fst::transition & tr) {
  return (tr.in() == "<E>" || tr.in().empty()) && tr.out().empty();
}




template<typename OutputIterator>
int lookup_subpatterns(const usyntagm_pattern & fst, OutputIterator out) {

  int nbtrans = 0;
  for (int q = 0; q < fst.size(); ++q) {
    
    for (usyntagm_pattern::transitions::const_iterator tr = fst.trans_begin(q);
         tr != fst.trans_end(q); ++tr) {
      
      if (tr->in().type == synt_input_type::SUBCALL) {
        const string & name = boost::get<string>(tr->in().v);
        //cout << name << " " << flush;
        *out = name;
        ++out;
      }
      ++nbtrans;
    }
  }
  return nbtrans;
}



} // namespace anonymous





int main(int argc, char ** argv) try {

  fs::path axiom, lingdefpath;

  {
    char * text = getenv("LINGDEF");
    if (text) {
      lingdefpath = fs::path(text, fs::native);
    }
  }


  argv++, argc--;
  if (argc == 0) { usage(); }

  while (argc) {
  
    string arg = *argv;

    if (arg[0] == '-') {

      if (arg == "-l") {

        argv++, argc--;
        if (argc == 0) { arg_needed(arg); }

        lingdefpath = fs::path(*argv, fs::native);

      } else if (arg == "-h") {

        usage();

      } else { unknown_arg(arg); }

    } else {
      axiom = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (axiom.empty() || lingdefpath.empty()) { bad_args(); }

  ling_def * lingdef = new ling_def(lingdefpath); 

  ugrammar gram(lingdef);


  fs::path dir(axiom.branch_path());

  string maingraph = fs::basename(axiom);

  vector<string> stack;
  stack.push_back(maingraph);

  int totalstates = 0, totaltrans = 0;

  while (! stack.empty()) {

    string name = stack.back();
    stack.pop_back();
  
    if (gram.get_syntagm_idx(name) != -1) { continue; } // already proceeded

    cout << "processing " << name << "... " << flush;
  
    fs::path grfpath = dir / (name + ".grf8");
  
    if (! fs::exists(grfpath)) {
      cerr << "error: unable to find " << grfpath.string() << endl;
      continue;
    }

    unitex_grf grf(grfpath);
    generic_fst fst;

    grf_to_generic_fst(grf, fst);

    cout << "flatten. " << flush;
    flatten(fst, grfpath.branch_path());

    cout << "prune. " << flush;
    fsa_prune(fst);

#warning "replace with fst_remove_epsilon ..."
    cout << "e-rem. " << flush;
    fsa_remove_epsilon(fst, is_epsilon);

    cout << "deter. " << flush;
    fsa_determinize(fst);

    cout << "minim. " << flush;
    fsa_minimize(fst);


    usyntagm_pattern pattern(fst, lingdef);
    pattern.name = name;

    int nbtrans = lookup_subpatterns(pattern, back_inserter(stack));
    int nbstate = pattern.size();

    gram.add_pattern(pattern);

    totalstates += nbstate;
    totaltrans  += nbtrans;

    cout << "done (" << nbstate << " states, " << nbtrans << " trans)\n";
  }


  fs::path grampath = fs::change_extension(axiom, ".ugram");

  cout << "writing grammar into " << grampath.string() << '\n'
    << gram.size() << " patterns\n"
    << "total of " << totalstates << " states and " << totaltrans << " transitions.\n";

  gram.write(grampath);
  cout << "done.\n";

  return 0;

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

