#include <iostream>
#include <fstream>
#include <queue>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/lexical_cast.hpp>

#include <cstdlib>

#include <outilex/lingdef.h>

#include <outilex/xml_text_uchart.h>
#include <outilex/uchart.h>
#include <outilex/featstruct.h>
#include <outilex/fs_node.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;


namespace {

char * progname;

enum match_type { 
  MATCH_ALL,     // dump all parse which are affix of the sentence
  MATCH_PREFIX,  // dump all parse which are prefix of the sentence
  MATCH_WHOLE    // dump all parse which cover the whole sentence
} match_mode = MATCH_WHOLE;

bool check_wellformed = false;

void usage() {
  cout << 
    "\n"
    "usage: " << progname << " [options] <chart>\n"
    "\n"
    "output sentence which have not been successfully parsed.\n"
    "\n"
    "options:\n"
    " -l <lingdef>  : specify the tagset description file (look for LINGDEF env var by default)\n"
    " -all|-affix   : print all matches (whatever their position in the sentence) this is the default\n"
    " -init|-prefix : print only matches which are prefix of a sentence\n"
    " -whole        : print only matches which cover a whole sentence\n"
    " -wellformed   : show only parse whose f-structure is well formed\n"
    "\n";
  exit(0);
}

} // anonymous namespace




int nb_matches(const uchart & chart) {

  int num = 0;
  for (int q = 0; q < chart.size(); ++q) {

    for (uchart::const_synt_iterator it = chart.synt_begin(q); it != chart.synt_end(q); ++it) {

      if (it->name == "mainP") {

        /* check if synt arrives to a final synt (if in MATCH_WHOLE mode) */
        if ((match_mode == MATCH_WHOLE) && ! chart.fsa.final(it->to)) { continue; }

        /* check if featstruct is well formed */
        if (check_wellformed && ! is_wellformed(it->fs)) { continue; }

        ++num;
      }
    }

    if (match_mode != MATCH_ALL) { // print synts only from initial state
      break;
    }
  }

  return num;
}


int main(int argc, char ** argv) try {

  fs::path chartpath, lingdefpath;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {
    
    string arg = *argv;
 
    if (arg == "-h") {
    
      usage();
    
    } else if (arg == "-l") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args: '-l' needs an argument\n"; exit(1); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-all" || arg == "-affix") {

      match_mode = MATCH_ALL;

    } else if (arg == "-init" || arg == "-prefix") {
      
      match_mode = MATCH_PREFIX;

    } else if (arg == "-whole") {

      match_mode = MATCH_WHOLE;

    } else if (arg == "-wellformed") {

      check_wellformed = true;

    } else {
      chartpath = fs::path(arg, fs::native);
    }
    argv++, argc--;
  }

  if (lingdefpath.empty() || chartpath.empty()) {
    cerr << progname << ": arguments missing\n";
    exit(1);
  }


  ling_def lingdef(lingdefpath);

  unification_init(&lingdef);

  xml_itext_uchart ichart(chartpath, & lingdef);

  uchart chart;
  int nbmatch = 0, nbsentence = 0;


  ofstream match("matches");
  ofstream fail("fails");

  while (ichart >> chart) {
    if (nb_matches(chart) == 0) {
      fail << chart.fsa.text << '\n';
    } else {
      match << chart.fsa.text << '\n';
      nbmatch++;
    }
    ++nbsentence;
  }
  double coverage = (((double) nbmatch) / ((double) nbsentence)) * 100.;

  cout << nbmatch << "/" << nbsentence << " matches (" << coverage << "%)\n";
  return 0;

} catch (exception & e) {

  cerr << "fatal error: " << e.what() << endl;
  exit(1);

} catch (...) { cerr << "ouch!\n"; exit(1); }


