#include <iostream>
#include <vector>

#include <boost/tuple/tuple.hpp>

#include <outilex/sentence_fsa.h>
#include <outilex/syntref.h>
#include <outilex/transduction4.h>


using namespace std;

//namespace {

typedef string_output output_type;

struct match_res {

  int to;
  synt_path_type path;
  output_type out;

  match_res() : to(-1), path(), out() {}
  match_res(int q, const output_type & o) : to(q), path(), out(o) {}
  match_res(int q, const syntref & ref, const output_type & o) : to(q), path(1, ref), out(o) {}
  match_res(int q, const synt_path_type & p, const output_type & o) : to(q), path(p), out(o) {}

  bool operator<(const match_res & b) const {
  
    if (to != b.to) { // best match is the one matching the longest text segment
      return to < b.to;
    }
  
    if (out < b.out) {
      return true;
    } else if (b.out < out) {
      return false;
    }
  
    // if segment and and output are the same
    // best path is the one passing with compounds ... else no best path
  
    return path.size() > b.path.size();
  }


  match_res & operator+=(const match_res & b) {
    if (*this < b) {
      *this = b;
    }
    return *this;
  }

  // match res concatenation

  match_res & operator*=(const match_res & b) {
    to = b.to;
    path.insert(path.end(), b.path.begin(), b.path.end());
    out.set_delay(b.path.size());
    out *= b.out;
    return *this;
  }
};

ostream & operator<<(ostream & os, const match_res & m) {
  os << "[ " << m.to << "; " << m.path << " ; " << m.out << " ]";
  return os;
}

struct cache_res {

  typedef my_deter_string_fst fst_type;

  typedef map<pair<int, int>, int> states_map;

  const sentence_fsa & text;
  my_deter_string_fst & T;

  struct state {
    match_res res;
    bool done;
    state() : res(), done(false) {}
  };
 
  vector<state> states;
  states_map    states_id;

  cache_res(sentence_fsa & fsa, my_deter_string_fst & T_)
    : text(fsa), T(T_), states(), states_id() {}


  inline match_res & res(int q) { return states[q].res; }

  inline bool & done(int q) { return states[q].done; }


  int find_state(const pair<int, int> & id) const {
    states_map::const_iterator it = states_id.find(id);
    if (it == states_id.end()) { return -1; }
    return it->second;
  }

  inline int find_state(int q1, int q2) const {
    return find_state(make_pair(q1, q2));
  }

  int add_state(int q1, int q2) {

    pair<int, int> id(q1, q2);

    int res = find_state(id);
    if (res != -1) { return res; }
  
    res = states.size();
    states.push_back(state());
 
    states_id[id] = res;
    return res;
  }

  int inter_state(int q1, int q2);
};



#if 0
int cache_res::inter_state(int q1, int q2) {

  cerr << "interstate(" << q1 << ", " << q2 << ")\n";

  bool done;
  int q = add_state(q1, q2, done);
 
  if (done) {
    cerr << "already computed\n";
    return q; 
  } // already computed


  if (T.final(q2)) {
    cerr << q2 << " is FINAL!\n";
    assert(! T.final_outputs(q2).empty());
    res(q) = match_res(q1, *(T.final_outputs(q2).rbegin()));
  }

  // iterate on transition in text
  
  for (int transno = 0; transno < text.trans_size(q1); ++transno) {

    const sentence_fsa::transition & tr = text.get_trans(q1, transno);
    cerr << "label = " << tr.label() << endl;

    for (fst_type::const_trans_iterator str = T.trans_begin(q2); str != T.trans_end(q2); ++str) {
    
      cerr << "str label = " << str->in() << endl;

      if (str->in().match(tr.in())) {

        int to = inter_state(tr.to(), str->to());

        if (res(to).to != -1) {
          match_res tmpres(tr.to(), syntref(q1, transno), str->out());
          tmpres *= res(to);
          res(q) += tmpres;
        }
      }

      cerr << "next.\n";
    }
  }
  return q;
}
#endif


int cache_res::inter_state(int q1, int q2) {

  //cerr << "\ninter_state(" << q1 << ", " << q2 << ")\n";

  vector<pair<int, int> > stack;

  stack.push_back(make_pair(q1,q2));

  while (! stack.empty()) {

    boost::tie(q1, q2) = stack.back();

    //cerr << "process with (" << q1 << ", " << q2 << ")\n";
    //cerr << "stack size = " << stack.size() << endl;

    int q = add_state(q1, q2);
    
    if (done(q)) { stack.pop_back(); continue; }

    if (T.final(q2)) {
      //cerr << "MATCH!\n";
      assert(! T.final_outputs(q2).empty());
      // we take the last final output, which is supposed to be the best one ?
      res(q) += match_res(q1, syntref(q1, -1), *(T.final_outputs(q2).rbegin()));
    }
 
    bool ok = true;

    // iterate on transition in text
    for (int transno = 0; transno < text.trans_size(q1); ++transno) {

      const sentence_fsa::transition & tr = text.get_trans(q1, transno);

      //cerr << "lex = " << tr.in() << endl;

      for (fst_type::const_trans_iterator str = T.trans_begin(q2); str != T.trans_end(q2); ++str) {

        //cerr << "mask = " << str->in() << endl;

        if (str->in().match(tr.in())) {

          //cerr << tr.in() << " matches with " << str->in() << endl;

          int to = find_state(tr.to(), str->to());
          
          if (to != -1) {
          
            assert(done(to));

            if (ok && res(to).to != -1) {
              match_res tmpres(tr.to(), syntref(q1, transno), str->out());
              tmpres *= res(to);
              res(q) += tmpres;
            }
 
          } else { // push new state
            ok = false;
            stack.push_back(make_pair(tr.to(), str->to()));
          }
          break; // trans are determinized
        }
      }
    }

    if (ok) { done(q) = true; stack.pop_back(); }
  }

//  cerr << "out of inter_state(" << q1 << ", " << q2 << ") res = " << find_state(q1, q2) << "\n\n";
  return find_state(q1, q2);
}






void output_res(const sentence_fsa & text, const match_res & res, ostream & os, int transmode,
                int outputmode) {

  
  if (outputmode == HTML_OUTPUT) {
    os << "<strong>";
  }

  int size = res.path.size();

  if (transmode == IGNORE_MODE) {

    for (int i = 0; i < size - 1; ++i) {
      const syntref & ref = res.path[i];
      os << text.get_trans(ref.qno, ref.transno).in().form << ' ';
    }

  } else { // soit MERGE soit REPLACE

    assert(res.out != output_type::zero());

    string outputs[size];

    for (map<int, string>::const_iterator it = res.out.v.begin(), end = res.out.v.end();
         it != end; ++it) {
      assert(it->first < size);
      outputs[size - 1 - it->first] = it->second;
    }

    if (transmode == REPLACE_MODE) {
    
      for (int i = 0; i < size; ++i) {
        if (! outputs[i].empty()) { os << outputs[i] << ' '; }
      }

    } else { // MERGE_MODE

      for (int i = 0; i < size - 1; ++i) {
        if (! outputs[i].empty()) { os << outputs[i] << ' '; }
        const syntref & ref = res.path[i];
        os << text.get_trans(ref.qno, ref.transno).in().form << ' ';
      }

      if (! outputs[size -1].empty()) {
        os << outputs[size-1] << ' ';
      }
    }
  }


  if (outputmode == HTML_OUTPUT) {
    os << "</strong>";
  }
}


void output_trans(const sentence_fsa::transition & tr, ostream & os, int transmode,
                  int outputmode) {
  os << tr.in().form << ' ';
}

struct compare_dest {
  template<typename transition>
  bool operator()(const transition & a, const transition & b) {
    return a.to() < b.to();
  }
};

//}; // namespace ""



int transduction4(sentence_fsa & text, my_deter_string_fst & T, ostream & os,
                   int transmode, int outputmode) {

  int nbmatches = 0;

  cache_res cache(text, T);

  int q = 0;
  while (! text.final(q)) {

    int idx = cache.inter_state(q, T.start());
  
    const match_res & res = cache.res(idx);
  
    if (res.to != -1) { // we find a matching sequence
    
      //      cerr << "MATCH! to = " << res.to << "\n";
      ++nbmatches;
      output_res(text, res, os, transmode, outputmode);
      q = res.to;
    
    } else { // output label of the first transition
    
 
      // we choose the transition with the closest destination
      // to avoid passing through compounds and missing some matches

      sentence_fsa::const_trans_iterator tr = min_element(text.trans_begin(q),
                                                          text.trans_end(q), compare_dest());
      if (tr == text.trans_end(q)) {
        os << "*ERROR*\n";
        cerr << "ERROR: bad text automaton\n";
        return nbmatches;
      }

      output_trans(*tr, os, transmode, outputmode);
      q = tr->to();
    }
  }

  if (q < text.size() - 1) { cerr << "error: sentence fsa not topologicaly sorted\n"; }

  if (outputmode == HTML_OUTPUT) {
    os << "\n<br/>\n";
  } else {
    os << "\n";
  }
  return nbmatches;
}

