#ifndef _DETER_PRED_TRANSDUCER_H_
#define _DETER_PRED_TRANSDUCER_H_

#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/lexical_cast.hpp>
#include <algorithm>
#include <iostream>
#include <stdexcept>
#include <string>
#include <sstream>
#include <vector>
#include <list>
#include <set>

#include <outilex/xml.h>


template<typename PredFst>
class deter_pred_fst {

public:

  typedef PredFst pred_fst;

  //  typedef typename pred_fst::elem_input_type elem_input_type;
  typedef typename pred_fst::input_type input_type;
  typedef typename pred_fst::output_type output_type;

  //typedef typename pred_fst::bladata_type bladata_type;

  typedef std::set<output_type> outputs_type;
 

  typedef typename pred_fst::transition ndeter_transition;
  typedef typename std::list<ndeter_transition> ndeter_transitions;


  struct position {

    int q;
    output_type delayed;

    position(xmlNodePtr node) { read_XML(node); }
    position(int q_, const output_type & out) : q(q_), delayed(out) {}

    bool operator<(const position & b) const {
      if (q != b.q) { return q < b.q; }
      return delayed < b.delayed;
    }

    /* XML Serialisation */
    void write_XML(xmlwriter & writer) const {
      writer.start_element("position");
      writer.write_attribute("q", boost::lexical_cast<std::string>(q));
      delayed.write_XML(writer);
      writer.end_element();
    }

    void read_XML(xmlNode * node) {

      char * text = xmlGetProp(node, "q");
      q = boost::lexical_cast<int>(text);
      xmlFree(text);

      node = node->xmlChildrenNode;
      while (node) {
        if (xmlStrcmp(node->name, output_type::xml_name()) == 0) {
          delayed.read_XML(node);
          return;
        }
        node = node->next;
      }
    }
  };

  typedef std::set<position> stateid;



  struct transition {
  
    int to_;
    input_type in_;
    output_type out_;

    transition() : to_(-1), in_(), out_() {}
    transition(const input_type & in, const output_type & out, int to)
      : to_(to), in_(in), out_(out) {}

    int to() const { return to_; }

    input_type & in() { return in_; }
    const input_type & in() const  { return in_; }

    output_type & out() { return out_; }
    const output_type & out() const { return out_; }
  };

  typedef std::vector<transition> transitions;
  typedef typename transitions::iterator trans_iterator;
  typedef typename transitions::const_iterator const_trans_iterator;

  struct state {

    bool final;
    outputs_type final_outputs;

    struct info {

      outputs_type delayeds;
      ndeter_transitions ndeter_trans;

      info() : delayeds(), ndeter_trans() {}
    };

    typedef std::map<int, info> info_map;
    info_map infos;

    // determinized transitions
    transitions trans; 
  };


  typedef std::map<stateid, int> state_map;


public:

  deter_pred_fst() : A(), states(), id2state() {}

  void init(const pred_fst & fst) {
    clear();
    A = fst;
    stateid id;
    id.insert(position(A.start(), output_type::one()));
    add_state(id);
  }


  void clear() { A.clear(); states.clear(); id2state.clear(); }

  int start() const { return 0; }


  // this is where the magic takes place
  template<typename Atom>
  const_trans_iterator find_matching_trans(int from, const Atom & e) {

    //cerr << "find_matching_trans(" << get_name() << ", " << from << ", " << e << ")\n";

    state & Q = states[from];

    /* first lookup in determinized transitions */

    {
      trans_iterator tr = std::find_if(trans_begin(from), trans_end(from), match_entry<Atom>(e));
      if (tr != trans_end(from)) {
        return tr;
      }
    }

    /* extract matching trans,
     * and store in label the bigger lexical_mask matching with e and disjoint
     * with all labels of no matching transitions
     */

    typedef std::vector<std::pair<int, ndeter_transitions> > trans_vec;
    trans_vec matching_trans;

    input_type label; // == LEXIC
    output_type prefix = output_type::zero();

    int nb_ndeter_trans = 0;

    for (typename state::info_map::iterator it = Q.infos.begin(); it != Q.infos.end(); ++it) {

      int q = (*it).first;
      ndeter_transitions & ntrans = (*it).second.ndeter_trans;
      const std::set<output_type> & delayeds = (*it).second.delayeds;

      int no = matching_trans.size();
      matching_trans.resize(no + 1);

      matching_trans[no].first = q;
      ndeter_transitions & mtrans = matching_trans[no].second;

      nb_ndeter_trans += extract_matching_trans(e, ntrans, front_inserter(mtrans), label, delayeds, prefix);

      if (mtrans.empty()) { // no matching trans leaving from q
        matching_trans.pop_back();
      }
    }


    // label == label of the determinist transition
    // prefix == longer common prefix of matching transitions (with consideration for delayed outputs)

    if (matching_trans.empty()) { // no matching transition at all

      // ifthere is no more non deter trans, we don't bother to add a new deter transition
      if (nb_ndeter_trans == 0) {
        return Q.trans.end();
      }

      /* label is the lexical mask matching with e and disjoint with
       * all masks in ndeter trans.
       * we cut label so that it becomes disjoint with all mask in deter trans (but still matching e)
       * then we add a trans to deter trans, labelled with label and pointing to state -1
       */

      //int nb_fake_trans = 0;

      for (trans_iterator tr = Q.trans.begin(); tr != Q.trans.end(); ++tr) { // for each trans in deter trans

        /*
        if (tr->to() == -1) { // tr is a fake deter transition

          ++nb_fake_trans;

          // if the number of bad deter transition is greater than the number
          // of non deter transition we cleanup fake transitions.

          if (nb_fake_trans > nb_ndeter_trans) {
            //  cleanup_fake_trans(Q.trans);
            return Q.trans.end();
          }
        }
        */

        if (tr->to() != -1) { // if tr is a fake transition, we don't bother to cut label
          std::vector<input_type> ms;
          input_type::minus(label, tr->in(), back_inserter(ms));

          // find the label matching with e
          typename std::vector<input_type>::iterator it = find_if(ms.begin(), ms.end(), match_entry<Atom>(e));
          assert(it != ms.end());
          label = *it;
        }
      }
 

      // we had a new fake transitions

      //std::cerr << "adding fake trans label = " << label << std::endl;

      return Q.trans.insert(Q.trans.end(), transition(label, output_type::one(), -1));
    }


    /*
     *  dirty stuffs :
     *  * reinject subset of matching trans in ndeter trans
     *  * compute stateid on the fly
     */

    input_type m = label;
    std::vector<input_type> min;

    stateid to_id;


    /* we iterate through the matching trans in the invers order we found them */

    for (int no = matching_trans.size() - 1; no >= 0; --no) {

      //    cerr << "no = " << no << endl;

      int q = matching_trans[no].first;
      ndeter_transitions & ntrans = Q.infos[q].ndeter_trans;
      ndeter_transitions & mtrans = matching_trans[no].second;

      //    cerr << "trans from state " << q << endl;

      const outputs_type & delayeds = Q.infos[q].delayeds;

      for (typename ndeter_transitions::iterator itr = mtrans.begin(); itr != mtrans.end(); ++itr) {

        ndeter_transition & tr = *itr;

        assert(in(m, tr.in())); // m should be subset of tr.in) or m == in

        /* reinject (tr.in minus label) dans ndeter trans
         * (tr.in minus label) == (tr.in minus m) union (m minus label)
         * with (m minus label) == min
         */

        input_type::minus(tr.in(), m, back_inserter(min));
        // front_inserter ?
        std::copy(min.begin(), min.end(), ndeter_trans_inserter(front_inserter(ntrans), tr));

        m = tr.in();


        /* compute stateid */

        //     cerr << "compute stateid: prefix=" << prefix << "tr.out=" << tr.out() << endl;

        for (typename std::set<output_type>::const_iterator it = delayeds.begin(); it != delayeds.end(); ++it) {
          //cerr << "it=" << *it << endl;
          output_type mydelay = output_type::minus(output_type::mult(*it, tr.out()), prefix);
          //cerr << "mydelay=" << mydelay << endl;
          to_id.insert(position(tr.to(), mydelay));
        }
      }
    }

    //  cerr << "ok making new transition\n";

    int to = add_state(to_id);
    return add_trans(from, label, prefix, to);
  }



  bool & final(int q) { return states[q].final; }
  outputs_type & final_outputs(int q) { return states[q].final_outputs; }

  trans_iterator trans_end(int q) { return states[q].trans.end(); }


#if 0
  const xmlChar * xml_name() { return (const xmlChar *) xmlname_.c_str(); }


  void read_XML(xmlNode * node, bladata_type data) {

    clear();

    if (xmlStrcmp(node->name, A.xml_name()) == 0) {

      A.read_XML(node, data);

      // create initial state...
      stateid id;
      id.insert(position(A.start(), output_type::one()));
      add_state(id);
      // we're done
      return;
    }

    char * text = xmlGetProp(node, "size");
    if (text) {
      int size;
      std::istringstream(text) >> size;
      states.reserve(size);
      xmlFree(text);
    }

    node = node->xmlChildrenNode;

    int qno = 0;
    while (node) {

      if (xmlStrcmp(node->name, A.xml_name()) == 0) {

        //std::cerr << "A.xml_name\n";

        A.read_XML(node, data);

      } else if (xmlStrcmp(node->name, "state") == 0) {

        //std::cerr << "q = " << qno << std::endl;

        states.resize(qno + 1);
        states[qno].read_XML(node, data);
        qno++;

      } else if (xmlStrcmp(node->name, idmap_xml_name()) == 0) {

        //std::cerr << "idmap-xmlname\n";

        stateid id; int q;
        read_idmap_XML(node, id, q);
        id2state[id] = q; 
      }

      //std::cerr << "node next\n";
      node = node->next;
    }
  }

  void read_XML(const boost::filesystem::path & path, bladata_type data) {

    std::string fname(path.native_file_string());

    xmlDocPtr doc = xmlParseFile(fname.c_str());

    if (doc == NULL) {
      throw xml_parse_error("deter_fst::read_XML: file '" + fname + "' not successfully parsed.");
    }

    xmlNodePtr node = xmlDocGetRootElement(doc);

    if ((xmlStrcmp(node->name, xml_name()) != 0) && xmlStrcmp(node->name, A.xml_name()) != 0) {
      xmlFreeDoc(doc);
      throw xml_parse_error("XML_fst: read_XML: document " + fname + " of wrong type");
    }

    try {
      read_XML(node, data);
    } catch (std::exception & e) {
      std::cerr << "deter_pred_fst::read: error while parsing fst file '"
        << fname << "' : " << e.what() << std::endl;
      xmlFreeDoc(doc);
      throw e;
    }

    xmlFree(doc);
  }

  void write_XML(const boost::filesystem::path & path) const {
    xmlwriter writer(path);
    writer.set_indent(true);
    writer.start_document();
    write_XML(writer);
    writer.end_document();
  }

  void write_XML(xmlwriter & writer) const {

    writer.start_element(xmlname_);

    A.write_XML(writer);

    for (int q = 0; q < states.size(); ++q) {
      states[q].write_XML(writer, q);
    }

    for (typename state_map::const_iterator it = id2state.begin(); it != id2state.end(); ++it) {
      write_idmap_XML(writer, it->first, it->second);
    }
    writer.end_element();
  }
#endif

protected:

  pred_fst A;

  std::vector<state> states;
  state_map id2state;


  template<typename OutputIterator>
  struct ndeter_trans_insert_iterator {
  
    OutputIterator it;
    ndeter_transition trans;

    ndeter_trans_insert_iterator(OutputIterator out, const ndeter_transition & t)
      : it(out), trans(t) {}

    ndeter_trans_insert_iterator & operator*()  { return *this; }
    ndeter_trans_insert_iterator & operator++() { ++it; return *this; }

    void operator=(const input_type & in) {
      trans.in() = in; *it = trans;
    }
  };

  template<typename OutputIterator>
  ndeter_trans_insert_iterator<OutputIterator>
  ndeter_trans_inserter(OutputIterator out, const ndeter_transition & t) {
    return ndeter_trans_insert_iterator<OutputIterator>(out, t);
  }

  template<typename Atom>
  struct match_entry {

    const Atom & e;
    match_entry(const Atom & entry) : e(entry) {}

    bool operator()(const input_type & m) { return input_type::match(e, m); }
    bool operator()(const deter_pred_fst::transition & trans) { return input_type::match(e, trans.in()); }
  };


  // delete from the container transition pointing to state -1

  void cleanup_fake_trans(transitions & trans) {
    std::cerr << "cleanup fake trans...\n";
    transitions tmp;
    for (trans_iterator tr = trans.begin(); tr != trans.end(); ++tr) {
      if (tr->to() != -1) { tmp.push_back(*tr); }
    }
    trans.swap(tmp);
  }


protected:

  template<typename Atom, typename OutputIterator>
  int extract_matching_trans(const Atom & e, ndeter_transitions & ntrans,
                              OutputIterator out, input_type & m,
                              const std::set<output_type> & delayeds, output_type & prefix) {

    typename ndeter_transitions::iterator itr;

    int nbtrans = 0;

    for (itr = ntrans.begin(); itr != ntrans.end();) {

      ndeter_transition & tr = *itr;

      if (input_type::match(e, tr.in())) { // curr transition matches with e

        //  cerr << "extract matches: " << tr.in() << "MATCHES" << e << "!\n";

        // on découpe le label in = (in inter m) u (in minus m)

        //m = inter(tr.in(), m);
        m &= tr.in();

        // on insert (in minus m) dans les ndeter_trans
        input_type::minus(tr.in(), m, ndeter_trans_inserter(std::inserter(ntrans, itr), tr));

        // on place (in inter m) dans out
        tr.in() = m;

        *out = tr;
        out++;

        /* compute longer common prefix of matchin trans */

        for (typename std::set<output_type>::const_iterator it = delayeds.begin(); it != delayeds.end(); ++it) {
          // cerr << "delayed=" << *it << endl;
          prefix = output_type::plus(prefix, output_type::mult(*it, tr.out()));
        }

        // suprime (in inter m) des ndeter trans

        itr = ntrans.erase(itr);

      } else { // tr doesn't match

        // compute (m minus tr.in)

        std::vector<input_type> ms;
        input_type::minus(m, tr.in(), back_inserter(ms));

        // reduce mask m so that it is disjoint with tr.in but still describing e

        typename std::vector<input_type>::iterator it = std::find_if(ms.begin(), ms.end(),
                                                                     match_entry<Atom>(e));
        if (it == ms.end()) {
          std::cerr << "fatal error: extract_matching_trans: e=" << e << std::endl;
          std::cerr << "m = " << m << " tr->in = " << tr.in() << std::endl;
          std::cerr << "m - tr.in = { ";
          for (typename std::vector<input_type>::iterator i = ms.begin(); i != ms.end(); ++i) {
            std::cerr << *i << ", ";
          }
          std::cerr << " }\n";
          throw std::runtime_error("internal error in extract_matching_trans: problem with minus?");
        }
        assert(it != ms.end());
        m = *it;

        itr++;
      }
      ++nbtrans;
    }

    return nbtrans;
  }


  /* protected */
  trans_iterator trans_begin(int q) { return states[q].trans.begin(); }




  void init_state(int q, const stateid & id) {

    states[q].final = false;

    //assert(Q.infos.empty());
    //assert(Q.synt_trans.empty());

    for (typename std::set<position>::const_iterator it = id.begin(); it != id.end(); ++it) {

      typename state::info & inf = states[q].infos[it->q];

      output_type delayed = it->delayed.delay(); // increment delayed output pos by one

      inf.delayeds.insert(delayed);

      if (inf.ndeter_trans.empty()) {
        inf.ndeter_trans.insert(inf.ndeter_trans.begin(), A.trans_begin(it->q), A.trans_end(it->q)); 
      }

      if (A.final(it->q)) {
        states[q].final = true;
        /* we do not delay final outputs (as we don't read no more input) */
        states[q].final_outputs.insert(it->delayed);
        
      }
    }
  }


  int add_state(const stateid & id) {
  
    typename state_map::iterator it = id2state.find(id);
    if (it != id2state.end()) { return (*it).second; }

    int res = states.size();
    states.resize(res + 1);
    id2state[id] = res;

    //std::cerr << "add_state: id = " << res << std::endl;
    init_state(res, id);
    return res;
  }


  trans_iterator add_trans(int from, const input_type & in, const output_type & out, int to) {
    return states[from].trans.insert(states[from].trans.end(), transition(in, out, to));
  }


#if 0
  const xmlChar * idmap_xml_name() { return (const xmlChar *) "state_id_map"; }

  void read_idmap_XML(xmlNodePtr node, stateid & id, int & q) {

    char * text = xmlGetProp(node, "q");
    std::istringstream(text) >> q;
    xmlFree(text);

    id.clear();

    node = node->xmlChildrenNode;

    while (node) {
      if (xmlStrcmp(node->name, "position") == 0) {
        id.insert(position(node));
      }
      node = node->next;
    }
  }

  void write_idmap_XML(xmlwriter & writer, const stateid & id, int q) const {

    writer.start_element("state_id_map");
    writer.write_attribute("q", boost::lexical_cast<std::string>(q));
    std::for_each(id.begin(), id.end(), XML_Writer(writer));
    writer.end_element();
  }
#endif

};



#endif

