#ifndef STRING_OUTPUT2_H
#define STRING_OUTPUT2_H

#include <string>
#include <set>

#include <outilex/foreach.h>

struct delayed_output_elem {

  delayed_output_elem(const std::string & txt, int d) 
    : text(txt), delay(d) {}

  bool operator<(const delayed_output_elem & b) const {
    if (delay != b.delay) { return delay < b.delay; }
    return text < b.text;
  }

  std::string text;
  int delay;
};


struct delayed_output {

  std::set<delayed_output_elem> val;


  delayed_output() : val() {}
  
  delayed_output(const std::string & txt, int delay = 0) : val() {
    val.insert(delayed_output_elem(txt, delay));
  }

  
  void delay(int d = 1) {
    foreach_(const delayed_output_elem & o, val) {
      (const_cast<delayed_output_elem &>(o)).delay += d; 
    }
  }


  bool empty() const { return val.empty(); }

  bool operator<(const delayed_output & b) const {
    return val < b.val;
  }

  // concatenation: set union
  delayed_output & operator*=(const delayed_output & b) {
    val.insert(b.val.begin(), b.val.end());
    return *this;
  }

  // longest common prefix (LCP) : set intersection

  delayed_output & operator+=(const delayed_output & b) {
    std::set<delayed_output_elem> res;
    std::set_intersection(val.begin(), val.end(),
                          b.val.begin(), b.val.end(),
                          std::inserter(res, res.end()));
    val.swap(res);
    return *this;
  }

  // strip prefix : set difference

  delayed_output & operator-=(const delayed_output & b) {
    std::set<delayed_output_elem> res;
    std::set_difference(val.begin(), val.end(),
                        b.val.begin(), b.val.end(),
                        std::inserter(res, res.end()));
    val.swap(res);
    return *this;
  }

};

inline std::ostream & operator<<(std::ostream & os, const delayed_output_elem & e) {
  os << "('" << e.text << "', " << e.delay << ")";
  return os;
}
inline std::ostream & operator<<(std::ostream & os, const delayed_output & out) {
  os << "{ ";
  foreach_(const delayed_output_elem & e, out.val) { os << e; }
  os << " }";
  return os;
}

inline delayed_output operator*(const delayed_output & a, const delayed_output & b) {
  delayed_output res(a);
  res *= b;
  return res;
}

#endif
