#ifndef _XML_FST_H_
#define _XML_FST_H_

#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <set>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>

#include <outilex/xml.h>


/* template class which implement a generic FST serialisable into XML format */


template<typename InputType,
  typename OutputType,
  typename DataType> /* = ling_def * */
class XML_fst {

public:

  typedef XML_fst<InputType, OutputType, DataType> my_type;

  typedef InputType  input_type;
  typedef OutputType output_type;
  typedef DataType   bladata_type;


  struct label_type {
    input_type in;
    output_type out;

    label_type() : in(), out() {}
    label_type(const input_type & in_, const output_type & out_)
      : in(in_), out(out_) {}
  };
  

  typedef std::set<output_type> outputs_type;
  typedef outputs_type data_type;

  struct transition {
  
    int         to_;
    label_type label_;


    /* constructors */

    transition() : to_(-1), label_() {}

    template<typename Transition>
    transition(const Transition & tr) : to_(tr.to()), label_(tr.in(), tr.out()) {}

    transition(xmlNode * node, bladata_type data) { read_XML(node, data); }
    transition(const label_type & l, int to) : to_(to), label_(l) {}
    transition(const input_type & in, const output_type & out, int to)
      : to_(to), label_(in, out) {}

    /* accessor/setter */

    int & to() { return to_; }
    int to() const { return to_; }
 
    label_type & label() { return label_; }
    const label_type & label() const { return label_; }

    input_type & in() { return label_.in; }
    const input_type  & in()  const { return label_.in; }

    output_type & out() { return label_.out; }
    const output_type & out() const { return label_.out; }


    /* XML serialisation */

    static const xmlChar * xml_name() { return (const xmlChar *) "transition"; }

    void write_XML(xmlwriter & writer) const {
      writer.start_element("transition");
      writer.write_attribute("to", boost::lexical_cast<std::string>(to_));
      writer.start_element("in");
      label_.in.write_XML(writer);
      writer.end_element();
      writer.start_element("out");
      label_.out.write_XML(writer);
      writer.end_element();
      writer.end_element();
    }



    void read_XML(xmlNode * node, bladata_type data) {

      char * text = xmlGetProp(node, "to");
      std::istringstream(text) >> to_;
      xmlFree(text);

      node = node->xmlChildrenNode;

      while (node) {

        if (xmlStrcmp(node->name, "in") == 0) {

          xmlNodePtr cur = node->xmlChildrenNode;

          while (cur) {
            if (xmlStrcmp(cur->name, input_type::xml_name()) == 0) {
              label_.in.read_XML(cur, data);
              break;
            }
            cur = cur->next;
          }

        } else if (xmlStrcmp(node->name, "out") == 0) {

          xmlNodePtr cur = node->xmlChildrenNode;

          while (cur) {
            if (xmlStrcmp(cur->name, output_type::xml_name()) == 0) {
              label_.out.read_XML(cur);
              break;
            }
            cur = cur->next;
          }
        }
        node = node->next;
      }
    }
  };

  typedef std::vector<transition> transitions;
  typedef typename transitions::iterator trans_iterator;
  typedef typename transitions::const_iterator const_trans_iterator;



  struct state {

    bool final;
    outputs_type final_outputs;

    transitions trans;


    state() : final(false), final_outputs(), trans() {}

    void clear() { final = false; final_outputs.clear(); trans.clear(); }


    /* XML serialisation */

    static const xmlChar * xml_name() { return (const xmlChar *) "state"; }

    void write_XML(xmlwriter & writer, int id) const {
      writer.start_element("state");
      writer.write_attribute("id", boost::lexical_cast<std::string>(id));
      if (final) {
        writer.write_attribute("final", "1");
      }
      write_outputs_XML(writer, final_outputs);
      std::for_each(trans.begin(), trans.end(), XML_Writer(writer));
      writer.end_element();
    }


    void read_XML(xmlNodePtr node, bladata_type data) {

      clear();

      char * text = xmlGetProp(node, "final");

      if (text) {
        std::istringstream(text) >> final;
      } else { final = false; }

      xmlFree(text);

      node = node->xmlChildrenNode;
      while (node) {

        if (xmlStrcmp(node->name, "transition") == 0) {

          int size = trans.size();

          try {

            trans.resize(size + 1);
            trans[size].read_XML(node, data);

          } catch (xml_parse_error & e) {

            std::cerr << "XML_fst: unable to load transition: " << e.what() << '\n';
            trans.resize(size);
          }
        } else if (xmlStrcmp(node->name, "outputs") == 0) {
        
          read_outputs_XML(node, final_outputs);
        }

        node = node->next;
      }
    }

  protected:

    static const xmlChar * outputs_xml_name() { return (const xmlChar *) "outputs"; }

    static void write_outputs_XML(xmlwriter & writer, const outputs_type & out) {
      writer.start_element("outputs");
      std::for_each(out.begin(), out.end(), XML_Writer(writer));
      writer.end_element();
    }

    static void read_outputs_XML(xmlNode * node, outputs_type & out) {
      out.clear();
      node = node->xmlChildrenNode;
      while (node) {
        if (xmlStrcmp(node->name, output_type::xml_name()) == 0) {
          out.insert(output_type(node));
        }
        node = node->next;
      }
    }
  };


  /* FST constructor */

  XML_fst(const std::string & xmlname) : states(), xmlname_(xmlname) {}

  XML_fst(const std::string & xmlname, const boost::filesystem::path & path, bladata_type data)
    : states(), xmlname_(xmlname) { read(path, data); }

  XML_fst(const std::string & xmlname, xmlNodePtr node, bladata_type data)
    : states(), xmlname_(xmlname) { read_XML(node, data); }


  int start() const { return 0; }

  void set_final(int q, bool final = true) { states[q].final = final; }
  bool final(int q) const { return states[q].final; }

  void set_data(int q, const data_type & data) { states[q].final_outputs = data; }
  const data_type & data(int q) const { return states[q].final_outputs; }

  void merge_data(int q, const data_type & data) {
    states[q].final_outputs.insert(data.begin(), data.end());
  }
  
  const outputs_type & final_outputs(int q) const { return states[q].final_outputs; }
  outputs_type & final_outputs(int q) { return states[q].final_outputs; }

  int size() const { return states.size(); }

  void clear() { states.clear(); }

  int add_state() {
    int res = states.size();
    states.resize(res + 1);
    return res;
  }

  void add_trans(int from, const input_type & in, const output_type & out, int to) {
    states[from].trans.push_back(transition(in, out, to));
  }

  void add_trans(int from, const label_type & label, int to) {
    states[from].trans.push_back(transition(label, to));
  }


  /* transitions iteration */

  trans_iterator trans_begin(int q) { return states[q].trans.begin(); }
  const_trans_iterator trans_begin(int q) const { return states[q].trans.begin(); }

  trans_iterator trans_end(int q) { return states[q].trans.end(); }
  const_trans_iterator trans_end(int q) const { return states[q].trans.end(); }


  /* XML serialisation */

  const xmlChar * xml_name() { return (const xmlChar *) xmlname_.c_str(); }

  /*
  void read_XML(const boost::filesystem::path & path, bladata_type data);
  void read_XML(xmlNodePtr node, bladata_type data);
  void write_XML(xmlwriter) const;
  void write_XML(const boost::filesystem::path & path) const;
  */



  /* implementation details */

  void write_XML(xmlwriter & writer) const {
    writer.start_element(xmlname_);
    writer.write_attribute("size", boost::lexical_cast<std::string>(states.size()));

    for (int q = 0; q < states.size(); ++q) {
      states[q].write_XML(writer, q);
    }
    writer.end_element();
  }


  void write(const boost::filesystem::path & path) const {
    xmlwriter writer(path);
    writer.set_indent(true);
    writer.start_document();
    write_XML(writer);
    writer.end_document();
  }


  void read_XML(xmlNode * node, bladata_type data) {

    clear();

    char * text = xmlGetProp(node, "size");
    if (text) {
      int size;
      std::istringstream(text) >> size;
      states.reserve(size);
      xmlFree(text);
    }

    node = node->xmlChildrenNode;

    int qno = 0;
    while (node) {
      if (xmlStrcmp(node->name, "state") == 0) {
        states.resize(qno + 1);
        states[qno].read_XML(node, data);
        qno++;
      }
      node = node->next;
    }
    //std::cerr << "XML_fst loaded (" << size() << " states)\n";
  }


  void read(const boost::filesystem::path & path, bladata_type data) {

    std::string fname(path.native_file_string());

    xmlDocPtr doc = xmlParseFile(fname.c_str());

    if (doc == NULL) {
      throw xml_parse_error("XML_fst::read_XML: file '" + fname + "' not successfully parsed.");
    }

    xmlNodePtr node = xmlDocGetRootElement(doc);

    if (xmlStrcmp(node->name, xml_name()) != 0) {
      xmlFreeDoc(doc);
      throw xml_parse_error("XML_fst: read_XML: document " + fname + " of wrong type");
    }

    try {
      read_XML(node, data);
    } catch (std::exception & e) {
      std::cerr << "XML_fst::read: error while parsing fst file '"
        << fname << "' : " << e.what() << std::endl;
      xmlFreeDoc(doc);
      throw;
    }
    xmlFree(doc);
  }


  void swap(my_type & T) {
    states.swap(T.states);
    xmlname_.swap(T.xmlname_);
  }

protected:
  std::vector<state> states;
  std::string xmlname_;
};


#endif
