#include <iostream>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/lingdef.h>
#include <outilex/string_transducer.h>
#include <outilex/generic_fst.h>
#include <outilex/epsilon_removal.h>
#include <outilex/fsa-prune.h>
#include <outilex/fsa_nb_transitions.h>

#include <outilex/usage.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;

const char * USAGE_DESCRIPTION =
"usage: gfst2sfst [-l lingdef] [-r] <gfst>\n"
"\n"
"with\n"
" -l <lingdef> : specified the lingdef description file\n"
" -r           : remove epsilon transitions\n"
"\n"
"Compile a generic fst into a string transducer file.\n";

namespace {

struct is_epsilon {

  pos_def * pos_epsilon;

  is_epsilon(pos_def * pos) : pos_epsilon(pos) {}
  bool operator()(const lexical_mask & m) const { return m.get_pos() == pos_epsilon; }
};


} // namespace ""


int main(int argc, char ** argv) try {

  fs::path lingdefpath, ipath, opath;
  bool remove_epsilon = false;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text);
  }

  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {

      argv++, argc--;
      if (argc == 0) { bad_args(); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-r") {

      remove_epsilon = true;

    } else if (arg == "-h") {
    
      usage();
    
    } else {
      ipath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }


  if (lingdefpath.empty() || ipath.empty()) { bad_args(); }

  ling_def lingdef(lingdefpath);

  opath = fs::change_extension(ipath, ".sfst");

  generic_fst gfst(ipath);
  string_transducer sfst(gfst, & lingdef);

  if (remove_epsilon) {
    cerr << "epsilon removal\n";
    fst_remove_epsilon(sfst, is_epsilon(lingdef.epsilon_pos()),
                       fst_output_ops<string_transducer>::type());
    fsa_prune(sfst);
  }
  sfst.write(opath);

  cout << "done. result in " << opath.string() << " (" << sfst.size() << " states, "
    << fsa_nb_trans(sfst) << " transitions)\n";


} catch (exception & e) {
  cerr << "fatal exception : " << e.what() << endl;
  exit(1);
} catch (...) {
  cerr << "ouch!\n"; exit(1);
}

