#include <deque>
#include <list>

#include <boost/timer.hpp>
#include <boost/progress.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <boost/function_output_iterator.hpp>

#include <outilex/deter_string_fst.h>


using namespace std;
using namespace boost;


typedef deter_string_fst::input_type  input_type;
typedef deter_string_fst::output_type output_type;
typedef deter_string_fst::transition  transition;
typedef deter_string_fst::transitions transitions;
typedef deter_string_fst::stateid     stateid;
typedef deter_string_fst::position    position;

std::ostream & operator<<(std::ostream & os, const deter_string_fst::transition & trans) {
  os << "(" << trans.in_;// << ", " << out_ << ", " << to_ << ")";
  return os;
}

namespace {


struct trans_maker {

  trans_maker(const transition & t) : tr(t) {}

  transition operator()(const input_type & i) {
    tr.in() = i; return tr;
  }

  transition tr;
};


template<typename OutputIterator>
struct trans_outputer {

  trans_outputer(const transition & t, const OutputIterator & o) : tr(t), out(o) {}

  void operator()(const input_type & i) {
    tr.in() = i;
    *out = tr; ++out;
  }

  transition tr;
  OutputIterator out;
};

template<typename OutputIterator>
trans_outputer<OutputIterator> make_trans_outputer(const transition & t,
                                                   const OutputIterator & o) {
  return trans_outputer<OutputIterator>(t, o);
}

/* insere une transitions t dans une liste trans de transitions 2 a2 disjoints
 * ou égaux
 * decoupe les trans si nécessaire
 */
void insert_trans(const transition & t, transitions & trans) {

  std::deque<transition> queue;
  list<transition> to_insert;
  
  to_insert.push_back(t);

  for (transitions::iterator tr1 = to_insert.begin(); tr1 != to_insert.end(); ++tr1) {

    transitions::iterator tr2 = trans.begin(), end = trans.end();

    input_type i; // intersection

    /* on itere jusqu'à la fin ou jusqu'à ce qu'on trouve un masque a decouper */
    while (tr2 != end) {
 
      if (tr1->in() == tr2->in()) {
       /* les masques sont égaux donc l'étiquette de tr1 est nécessairement
        * egale ou disjointe avec toutes les trans on arrete l'iteration
        */
        tr2 = end;
        break;
      }
    
      i = tr1->in();
      i &= tr2->in();

      if (i) { // etiquettes non disjointes et non égales on sort de la boucle
        break;
      }
      ++tr2;
    }

    if (tr2 != end) { //tr1 intersecte *tr2, decoupage
    
      input_type label = tr2->in();
    
      // on decoupe tr1, on met le resultat à la fin de to_insert
      lexical_mask2::minus(tr1->in(), i,
            make_function_output_iterator(make_trans_outputer(*tr1,
                                                              back_inserter(to_insert))));
      tr1->in() = i;


      /* on découpe tr2 dans trans
       * on conserve le resultat dans back pour les prochaines transitions
       */

      transitions back;
      lexical_mask2::minus(label, i,
            make_function_output_iterator(make_trans_outputer(*tr2,
                                                              back_inserter(back))));
 
 
      /* on remplace la trans courante avec le resultat du decoupage
       */
    
      tr2->in() = i;
      trans.insert(tr2, back.begin(), back.end());
 
      ++tr2;
      while (tr2 != trans.end()) {
      
        if (lexical_mask2::intersects(tr2->in(), label)) { // on remplace
          
          // assert(tr2->in() == label)
          
          foreach_(transition & t, back) { // remap etat destination
            t.to() = tr2->to();
          }
          tr2->in() = i;
          trans.insert(tr2, back.begin(), back.end());
        }
        ++tr2;
      }
    
    }
  }

  // insert les transitions decoupe dans trans
  trans.splice(trans.end(), to_insert);
}


/* "determinise" une liste de transitions
 * c'est a dire decoupe les label de trans de maniere à ce qu'ils
 * soient tous deux a deux disjoints ou égaux
 */

void deter_trans_list(transitions trans) {
  
  transitions res;

  while (! trans.empty()) {
    transition & tr = trans.front();
    insert_trans(tr, res);
    trans.pop_front();
  }
  trans.swap(res);
}



/* compute the stateid which is reached by reading label from the translist
 * all trans labeled with label are removed from list
 */

void NEXT(const input_type & label, transitions & trans, stateid & id) {

  transitions::iterator tr = trans.begin();

  while (tr != trans.end()) {
    if (tr->in() == label) {
      id.insert(position(tr->to(), tr->out()));
      tr = trans.erase(tr);
    } else { ++tr; }
  }
}

/* compute the longest common prefix of the delayed output in stateid
 * and erase it from those output
 */

static void LCP(stateid & id, output_type & prefix) {

  //  assert(stateid.size() != 0);
  
  prefix = id.begin()->delayed;

  for (stateid::iterator it = id.begin(); it != id.end(); ++it) {
    prefix += it->delayed;
  }

  if (! prefix.empty()) { //strip prefix
    
    stateid nid;

    foreach_(position pos, id) {
      pos.delayed -= prefix;
      nid.insert(nid.end(), pos);
    }
    id.swap(nid);
  }
}


transition trans1to2(const string_transducer::transition & t1) {
  deter_string_fst::transition t2(lexical_mask2(t1.in()),
                                  delayed_output(t1.out()), t1.to());
  return t2;
}

} //namespace ""


void deter_string_fst::init_fst(const string_transducer & T) {

  //cerr << "init_fst size = " << T.size() << "\n";

  boost::timer tmr;
  boost::progress_display progress(T.size(), std::cerr);

  states.clear();
  states.resize(T.size());

  for (int q = 0; q < T.size(); ++q) {

    //cerr << "q = " << q << "\n";
    stateid id;
    id.insert(position(q, delayed_output()));

    id2state[id] = q;

    set_final(q, T.final(q));

    //states[q].final_outputs = T.final_outputs(q);
    /*
    std::copy(T.final_outputs(q).begin(), T.final_outputs(q).end(),
              std::inserter(states[q].final_outputs,
                            states[q].final_outputs.begin()));
                            */
    foreach_(const string & txt, T.final_outputs(q)) {
      // delay -1 because in final output 
      states[q].final_outputs.insert(states[q].final_outputs.end(), delayed_output(txt, -1));
    }

    states[q].trans.insert(states[q].trans.begin(),
                           make_transform_iterator(T.trans_begin(q), trans1to2),
                           make_transform_iterator(T.trans_end(q), trans1to2));
#if 0
    cerr << "avant:\n";
    std::copy(states[q].trans.begin(), states[q].trans.end(),
              ostream_iterator<transition>(cerr, "\n"));
    cerr << "\n";
#endif
    deter_trans_list(states[q].trans);

#if 0
    cerr << "apres:\n";
    std::copy(states[q].trans.begin(), states[q].trans.end(),
              ostream_iterator<transition>(cerr, "\n"));
    cerr << "\n";
#endif

    ++progress;
  }
  std::cerr << "out of init_fst, " << tmr.elapsed() << "s.\n";
}



void deter_string_fst::init_state(int q, const stateid & id) const {

  states[q].flags = 0; // ni final, ni deter

  for (stateid::const_iterator it = id.begin(); it != id.end(); ++it) {
  
    delayed_output delayed = it->delayed;
#warning testing stuffs
   // delayed.delay();
  
    if (final(it->q)) {

      states[q].flags |= FINAL;

      if (! final_outputs(it->q).empty()) {

        for (outputs_type::iterator it2 = final_outputs(it->q).begin(),
             end = final_outputs(it->q).end();
             it2 != end; ++it2) {
        
          states[q].final_outputs.insert(delayed * *it2);
        }
      } else {
        states[q].final_outputs.insert(delayed);
      }
    }

    delayed.delay();

    transitions & trans = states[it->q].trans;

    foreach_(transition tr, trans) {
      tr.out() *= delayed; // * est une operation commutative ici
      insert_trans(tr, states[q].trans);
    }
  }
}



int deter_string_fst::add_state(const stateid & id) const {

  state_map::iterator it = id2state.find(id);
  if (it != id2state.end()) { return it->second; }

  int res = states.size();
  states.resize(res + 1);
  id2state[id] = res;

  init_state(res, id);
  return res;
}



void deter_string_fst::determinize(int q) const {


  if (states[q].flags & DETER) { return; }

  //cerr << "determinize(" << q << ")\n";

  transitions ndeter_trans;
  ndeter_trans.swap(states[q].trans);

  while (! ndeter_trans.empty()) {
  
    input_type label = ndeter_trans.front().in();
    //cerr << "label" = label << endl;

    stateid idto;
    NEXT(label, ndeter_trans, idto);

    output_type prefix;
    LCP(idto, prefix);

    int to = add_state(idto);
  
    states[q].trans.push_back(transition(label, prefix, to));
  }

  states[q].flags |= DETER;
  //cerr << "done.\n";
}


