#include <boost/filesystem/path.hpp>
#include <boost/lexical_cast.hpp>


#include <outilex/xml_format_lexic.h>
#include <outilex/xml_format_sentence_fsa.h>
#include <outilex/xml_inl_format_sentence_fsa.h>
#include <outilex/xml-names.h>
#include <outilex/sentence_fsa.h>
#include <iostream>
#include <outilex/xml_text_fsa.h>

using namespace std;
using namespace boost;
using namespace xmlnames;

namespace fs = boost::filesystem;

int LINK_WITH_XML_TEXT_FSA;

namespace {

// registration

bool xml_guess(const fs::path & path) try {

  xmlreader reader(path);
  
  reader.verbose_error(false);

  int ret;
  while (((ret = reader.read()) == 1) && (reader.node_type() != XML_READER_TYPE_ELEMENT)) {}
  reader.check();

  if (xmlStrcmp(reader.const_name(), TEXT_FSA_ELEM) != 0) { return false; }

  // cerr << "XML format detected\n";
  return true;

} catch (exception & e) {
  return false;
}


itext_fsa * new_xml_itext_fsa(const fs::path & path, ling_def * ldef) {
  return new xml_itext_fsa(path, ldef);
}

otext_fsa * new_xml_otext_fsa(const fs::path & path, int size, int compression) {
  return new xml_otext_fsa(path, size, compression);
}

text_fsa_reader_registerer ixml_registerer(xml_guess, new_xml_itext_fsa);
text_fsa_writer_registerer oxml_registerer("xml", new_xml_otext_fsa);

} // namespace ""

void xml_itext_fsa::open(const fs::path & p) {
  
  close();
  path = p;
  reader.open(path);

  int ret;
  while (((ret = reader.read()) == 1) && (reader.node_type() != XML_READER_TYPE_ELEMENT)) {}
  reader.check();

  if (xmlStrcmp(reader.const_name(), TEXT_FSA_ELEM) != 0) {
    cerr << "text_fsa::open : got an " << reader.const_name() << " instead of " << TEXT_FSA_ELEM << endl;
    throw xml_parse_error("text_fsa::open " + path.string() + ": bad document type"); 
  }

  char * text = reader.get_attribute(SIZE_ATTR);
  if (text) {
    size_ = lexical_cast<int>(text);
    xmlFree(text);
  } else { size_ = -1; }

  pos_ = 0;
}


void xml_itext_fsa::close() {
  path = fs::path();
  reader.close();
  size_ = pos_ = -1;
}



bool xml_itext_fsa::read_next(sentence_fsa & fsa) {

  if (! reader || pos_ == -1) {
    return false;
  }

  int ret = 1;

  do {
  
    if (reader.node_type() == XML_READER_TYPE_ELEMENT
        && xmlStrcmp(reader.const_name(), LEXIC_ELEM) == 0) { // new lexic
 
      xmlNode * node = reader.expand();
 
      lexic_read_xml(node, lexic, lingdef);
 
      ret = reader.next();
      reader.check();
 
    } else if (reader.node_type() == XML_READER_TYPE_ELEMENT
               && xmlStrcmp(reader.const_name(), SENTENCE_FSA_ELEM) == 0) {
    
      xmlNode * node = reader.expand();

      sentence_fsa_read_xml(node, fsa, lexic, lingdef);
    
      ret = reader.next(); // skip subtree
      reader.check();

      ++pos_;
      return true;

    } else { 
      /*
      cerr << "xml_read_fsa: ELSE:\n";
      cerr << "node_type = " << reader.node_type() << "name: " << reader.const_name() << endl;
      cerr << "pos = " << pos_ << endl;
      */
      ret = reader.read();
      //cerr << "ret = " << ret << endl;
    }
  
  } while (ret == 1);

  reader.check();

  assert(ret == 0);

  /* EOF */

  //cerr << "EOF reached pos = " << pos_ << endl;

  pos_ = -1;
  return false;
}

void xml_itext_fsa::rewind() { open(path); }

void xml_itext_fsa::seek(int offset) {

  if (! reader) { 
    throw runtime_error("xml_itext_fsa::seek: no XML stream");
  }

  if (offset < pos_) { rewind(); }

  int ret = 1;
  while (ret == 1 && pos_ < offset) {
    
    if (reader.node_type() == XML_READER_TYPE_ELEMENT
        && xmlStrcmp(reader.const_name(), LEXIC_ELEM) == 0) { // new lexic
 
      xmlNode * node = reader.expand();
 
      lexic_read_xml(node, lexic, lingdef);
 
      ret = reader.next();
      reader.check();
 
    } else if (reader.node_type() == XML_READER_TYPE_ELEMENT
               && xmlStrcmp(reader.const_name(), SENTENCE_FSA_ELEM) == 0) {

      ret = reader.next();
      ++pos_;
    
    } else { ret = reader.read(); }
  
    reader.check();
  }

  if (ret != 1) {
    pos_ = -1;
    throw runtime_error("xml_itext_fsa::seek: unable to seek (EOF)");
  }
}





/* xml_otext_fsa */


void xml_otext_fsa::open(const fs::path & path, int size, int compression) {

  close();

  writer.open(path, compression);
  writer.set_indent(1);

  writer.start_document();
  writer.start_element(TEXT_FSA_ELEM);

  if (size != -1) {
    writer.write_attribute(SIZE_ATTR, lexical_cast<string>(size));
  }
}


void xml_otext_fsa::close() {

  if (! writer) { return; }

  writer.end_element(); // text-fsa
  writer.end_document();
  writer.close();
}


#warning "a revoir"
void xml_otext_fsa::write(const sentence_fsa & fsa) {

  if (lexic != fsa.lexic) {
    //    cerr << "ofsa::write: change lexic!\n";
    lexic = fsa.lexic;
    lexic_write_xml(writer, lexic);
    //cerr << "lexic dumped: " << lexic.size()<< " elements.\n";
  }
  sentence_fsa_write_xml(writer, fsa);

 // sentence_fsa_write_inline_xml(writer, fsa);
}
