#ifndef _TEXT_FSA_H_
#define _TEXT_FSA_H_

#include <boost/filesystem/path.hpp>
#include <stdexcept>


class sentence_fsa;
class itext_fsa;
class otext_fsa;
class ling_def;

typedef bool (*ifsa_guesser_f)(const boost::filesystem::path & path);
typedef itext_fsa * (*ifsa_creator_f)(const boost::filesystem::path & path, ling_def * ldef);

typedef otext_fsa * (*ofsa_creator_f)(const boost::filesystem::path & path, int size, int compression);

void register_text_fsa_reader(ifsa_guesser_f guess, ifsa_creator_f create);
void register_text_fsa_writer(const std::string & format, ofsa_creator_f create);


struct text_fsa_reader_registerer {
  text_fsa_reader_registerer(ifsa_guesser_f guess, ifsa_creator_f create) {
    register_text_fsa_reader(guess, create);
  }
};

struct text_fsa_writer_registerer {
  text_fsa_writer_registerer(const std::string & format, ofsa_creator_f create) {
    register_text_fsa_writer(format, create);
  }
};
/** @brief Class itext_fsa that allows for the reading of a text fsa for each sentence
*/
class itext_fsa {

public:
  virtual ~itext_fsa();

  virtual int size() const { return 0; }

  virtual bool read_next(sentence_fsa & fsa) { return false; }

  virtual int tell() const { return 0; }
  virtual void seek(int pos) { throw std::runtime_error("seek not implemented"); }
  virtual void rewind() {}

  // on_lexic_change(...) ...
 
  virtual bool ok() const { return false; }

  operator const void*() const { return ok() ? this : 0; }
  bool operator!() const { return ! ok(); }
};

inline itext_fsa & operator>>(itext_fsa & itext, sentence_fsa & fsa) {
  itext.read_next(fsa); return itext;
}


/* return a generic text_fsa reader for the file specified in path 
 * the function guess the format and return the appropriate parser.
 * throw on error
 */

itext_fsa * new_itext_fsa(const boost::filesystem::path & path, ling_def * lingdef);

/* return true if the file is recognized as a text fsa */

bool is_text_fsa(const boost::filesystem::path & path);

/** @brief Class otext_fsa is used for printing a text fsa for each sentence
 */

class otext_fsa {

public:

  virtual ~otext_fsa();
  
  virtual void close() {}

  virtual void write(const sentence_fsa & fsa) {}

  virtual bool ok() const { return false; }

  operator const void*() const { return ok() ? this : 0; }
  bool operator!() const { return ! ok(); }
};

inline otext_fsa & operator<<(otext_fsa & text, const sentence_fsa & fsa) {
  text.write(fsa); return text;
}

/** return a new text_fsa formater for the specified format
 * the compression parameter is an hint 
 * (only use with the xml format for now)
 */

otext_fsa * new_otext_fsa(const boost::filesystem::path & path,
                          const std::string & format, int size, int compression = 0);

#endif
