#ifndef _SENTENCE_FSA_H_
#define _SENTENCE_FSA_H_

#include <boost/iterator/transform_iterator.hpp>
#include <boost/utility.hpp>

#include <string>
#include <vector>

#include <outilex/basic_fsa.h>
#include <outilex/lexic.h>


class ling_def;

  /** @brief class sentence_fsa is composed of the automaton repreenting all analyses of a sentence
   */

class sentence_fsa {

public: // data types

  /** @brief struct meta data associated to each state in the sentence fsa
   * contains the corespondent position in text
   */

  struct meta_data {

    int pos;

    meta_data() : pos(0) {}
    meta_data(const meta_data & m) : pos(m.pos) {}

    /* merge 2 pos, assert (pos ==m.pos) ? */
    meta_data & operator|=(const meta_data & m) { pos = m.pos; return *this; }
  };


  /* underline (sous-jacent?) fsa has trans labeled by int */
 
  typedef basic_fsa<int, meta_data> intern_fsa_type;

  typedef intern_fsa_type::transition intern_transition;

  typedef intern_fsa_type::trans_iterator intern_trans_iterator;
  typedef intern_fsa_type::const_trans_iterator intern_const_trans_iterator;


  // non const transition, can change 'to' member
  struct transition {

    const LEXIC * lex;
    intern_transition * tr;

    transition(const LEXIC * l, intern_transition & t) : lex(l), tr(&t) {} 

    int to() const { return tr->to(); }
    int & to() { return tr->to(); }

    int labelno() const { return tr->label(); }
    const lexical_mask & label() const { return (*lex)[tr->label()]; }
    const lexical_mask & in() const { return (*lex)[tr->label()]; }
  };


  struct const_transition {

    const LEXIC * lex;
    const intern_transition * tr;

    const_transition(const LEXIC * l, const intern_transition & t) : lex(l), tr(&t) {} 
    const_transition(const transition & tr) : lex(tr.lex), tr(tr.tr) { }

    int to() const { return tr->to(); }

    int labelno() const { return tr->label(); }
    const lexical_mask & label() const { return (*lex)[tr->label()]; }
    const lexical_mask & in() const { return (*lex)[tr->label()]; }
  };



  struct const_trans_wrapper
    : public std::unary_function<intern_transition, const_transition> {
  
      const LEXIC * lex;

      const_trans_wrapper(const sentence_fsa & fsa) : lex(&fsa.lexic) {}
      const_trans_wrapper(const LEXIC * l) : lex(l) {}

      const_transition operator()(const intern_transition & tr) const {
        return const_transition(lex, tr);
      }
  };

  struct trans_wrapper
    : public std::unary_function<intern_transition, transition> {
  
      const LEXIC * lex;

      trans_wrapper(sentence_fsa & fsa) : lex(&fsa.lexic) {}
      trans_wrapper(LEXIC * l) : lex(l) {}

      transition operator()(intern_transition & tr) const {
        return transition(lex, tr);
      }
  };


  typedef boost::transform_iterator<trans_wrapper, intern_trans_iterator> trans_iterator;
  typedef boost::transform_iterator<const_trans_wrapper,
          intern_const_trans_iterator> const_trans_iterator;


public: // data members

  intern_fsa_type A;
  std::string text;
  ling_def * lingdef;
  LEXIC lexic;


public: // constructors
   
  /** 
   * @brief constructor
   * 
   * @param ldef lingdef
   */

  sentence_fsa(ling_def * ldef = 0) : A(), text(), lingdef(ldef), lexic() {}


public: // modifiers

  /** 
   * @brief clear the sentence automaton
   * 
   */

  void clear() { A.clear(); text.clear(); }

  /** 
   * @brief set lingdef
   * 
   * @param ldef lingdef
   */
  void set_lingdef(ling_def * ldef) { lingdef = ldef; }

  void set_lexic(const LEXIC & lex) { lexic = lex; }

  void strip_lexic(mutable_lexic & mlex);
  void strip_lexic() { mutable_lexic mlex; strip_lexic(mlex); }

  void set_final(int q, bool final = true) { A.set_final(q, final); }

  void set_pos(int q, int pos) { A.data(q).pos = pos; }


  trans_iterator trans_begin(int q) { 
    return trans_iterator(A.trans_begin(q), trans_wrapper(*this));
  }

  trans_iterator trans_end(int q) { 
    return trans_iterator(A.trans_end(q), trans_wrapper(*this));
  }

  void swap(sentence_fsa & fsa) {
    A.swap(fsa.A); text.swap(fsa.text);
    std::swap(lingdef, fsa.lingdef);
    lexic.swap(fsa.lexic);
  }

  void determinize();
  void prune();
  void topological_sort();

public: // queries

  int size() const { return A.size(); }
  bool empty() const { return size() == 0; }

  bool final(int q) const { return A.final(q); }

  int pos(int q) const { return A.data(q).pos; }

  int trans_size(int q) const { return A.trans_size(q); }

  const_transition get_trans(int q, int transno) const {
    return const_transition(&lexic, A.get_trans(q, transno)); 
  }

  transition get_trans(int q, int transno) {
    return transition(&lexic, A.get_trans(q, transno)); 
  }

  const_trans_iterator trans_begin(int q) const { 
    return const_trans_iterator(A.trans_begin(q), const_trans_wrapper(*this));
  }
  const_trans_iterator trans_end(int q) const { 
    return const_trans_iterator(A.trans_end(q), const_trans_wrapper(*this));
  }
};

#endif
