#include <iostream>
#include <string>

#include <boost/filesystem/path.hpp>

#include <outilex/text_format_lexic.h>
#include <outilex/bin_format_sentence_fsa.h>
#include <outilex/sentence_fsa.h>
#include <outilex/serialize.h>

#include <outilex/bin_text_fsa.h>

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

int LINK_WITH_BIN_TEXT_FSA;

namespace {
const int MAGIC = 0xeB14eF5A;

// registration stuffs

bool bin_guess(const fs::path & path) {
  fs::ifstream f(path);
  if (! f) { return false; }
  int magic;
  read_int(f, magic);
  if (magic != MAGIC) { return false; }
  //  cerr << "BIN format detected\n";
  return true;
}

itext_fsa * new_bin_itext_fsa(const fs::path & path, ling_def * ldef) {
  return new bin_itext_fsa(path, ldef);
}

otext_fsa * new_bin_otext_fsa(const fs::path & path, int size, int compression = 0) {
  return new bin_otext_fsa(path, size);
}

text_fsa_reader_registerer ibin_registerer(bin_guess, new_bin_itext_fsa);

text_fsa_writer_registerer obin_registerer("bin", new_bin_otext_fsa);

} // namespace "" 


void bin_itext_fsa::open(const fs::path & p) {
  
  close();
  path = p;
  stream.open(path);

  int magic;
  read_int(stream, magic);

  if (magic != MAGIC) {
    cerr << "bin_itext::open : bad file format : no magic found\n";
    throw runtime_error("txt_open: " + p.string() + " read error\n");
  }

  read_int(stream, size_);

  //  cerr << "open ok, size = " << size_ << "\n";
  pos_ = 0;
}


void bin_itext_fsa::close() {
  if (stream.is_open()) {
    stream.close();
  }
  stream.clear();
  size_ = pos_ = -1;
}


bool bin_itext_fsa::read_next(sentence_fsa & fsa) {

//  cerr << "bin;read_next()\n";
  if (! stream || pos_ == -1) {
    cerr << "read error\n";
    return false;
  }

  char c;

  while (stream.get(c)) {

    if (c == 'L') { // lexic
 
      //cerr << "bin::read_next : new lexic";
      lexic_read_text(stream, lexic, lingdef); // lexic is just text
 
      //cerr << "lexic size = " << lexic.size() << '\n';

    } else if (c == 'F') { // fsa
    
      //      cerr << "fsa\n";
      sentence_fsa_read_bin(stream, fsa, lexic, lingdef);
      return true;

    } else {
      cerr << "bin_txt_fsa::read_next: bad file format";
      throw runtime_error("bin_txt_fsa: read: parse error");
    }
  }

  /* EOF */
 
  pos_ = -1;
  return false;
}


void bin_itext_fsa::rewind() { open(path); }

void bin_itext_fsa::seek(int offset) {

  if (! stream) {
    throw runtime_error("bin itext_fsa::seek: no stream");
  }

  if (pos_ == -1 || offset < pos_) { rewind(); }

  char c;
  while (pos_ < offset && stream.get(c)) {
    
    if (c == 'L') {
 
      lexic_read_text(stream, lexic, lingdef);
    
    } else if (c == 'F') {

      sentence_fsa_skip_bin(stream);
      ++pos_;
    }
  }

  if (!stream) { 
    pos_ = -1;
    throw runtime_error("bin_itext_fsa::seek : EOF");
  }
}




/* otext_fsa */

bin_otext_fsa::bin_otext_fsa(const boost::filesystem::path & path, int size)
  : stream(), lexic() { 

  stream.exceptions(std::ios_base::eofbit | std::ios_base::failbit | std::ios_base::badbit);
  open(path, size); 
}

void bin_otext_fsa::open(const fs::path & path, int size) {

  close();

  lexic.clear();
  stream.open(path);

  write_int(stream, MAGIC);
  write_int(stream, size);
}


void bin_otext_fsa::close() {
  if (stream.is_open()) { stream.close(); }
}


void bin_otext_fsa::write(const sentence_fsa & fsa) {

  if (lexic != fsa.lexic) {
    //cerr << "bin::write_next : new lexic\n";
    lexic = fsa.lexic;
    //cerr << "lexic size = " << lexic.size() << "\n";
    stream.put('L');
    lexic_write_text(stream, lexic);
  }

  stream.put('F');
  sentence_fsa_write_bin(stream, fsa);
}

