#include <iostream>
#include <stack>

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/convenience.hpp>
                       
#include <outilex/FST2_to_syntagm_pattern.h>
#include <outilex/lingdef.h>
#include <outilex/grammar.h>

using namespace std;

namespace fs = boost::filesystem;

template<typename Value>
struct stack_pusher {

  typedef Value value_type;
  typedef stack_pusher<value_type> my_type;

  stack<value_type> * stk;

  stack_pusher(stack<value_type> & s) : stk(& s) {}

  my_type & operator*() { return *this; }
  my_type & operator++() { return *this; }
  void operator++(int) {}

  template<typename V>
  void operator=(const V & v) { stk->push(v); }
};

template<typename Value>
inline stack_pusher<Value> make_stack_pusher(stack<Value> & s) {
  return stack_pusher<Value>(s);
}




template<typename OutputIterator>
int lookup_subpatterns(const syntagm_fst & synt_fst, OutputIterator out) {

  int res = 0;

  const syntagm_pattern & fst = synt_fst.get_pattern();

  int size = fst.size();
  for (int q = 0; q < fst.size(); ++q) {
    for (syntagm_pattern::const_synt_trans_iterator tr = fst.synt_trans_begin(q);
         tr != fst.synt_trans_end(q); ++tr) {
    
      cerr << "adding " << tr->in().get_name() << endl;
      *out = tr->in().get_name();
      ++out;
      res++;
    }
  }
  return res;
}


char * progname;


void usage() {
  cerr << "usage: " << progname << " -l <lingdef> <fstpattern>\n";
  exit(1);
}


int main(int argc, char ** argv) {

  string fstname, lingdefname;

  progname = *argv;

  argv++, argc--;

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefname = *argv;
    
    } else if (arg == "-h") {
    
      usage();
    
    } else {
      fstname = arg;
    }

    argv++, argc--;
  }

  if (fstname.empty() || lingdefname.empty()) { usage(); }


  try {

    ling_def * lingdef = new ling_def(lingdefname);

    cerr << endl;

    grammar grm(lingdef);

    fs::path fstpath(fstname, fs::native);

    fs::path grammarpath = fs::change_extension(fstpath, ".grm");

    fstpath = fs::change_extension(fstpath, ""); // strip .ft2 extension

    fs::path dir(fstpath.branch_path()); // retrieve the directory where are stored the grammars
    string maingraph = fstpath.leaf();

    stack<string> stk;
    stk.push(maingraph);

    while (! stk.empty()) {

      string name = stk.top();
      stk.pop();

      cout << "proceeding with " << name << "...\n";

      if (grm.get_syntagm_idx(name) != -1) { cout << "already proceeded\n"; continue; }

      fs::path fst_path(dir / (name + ".fst2"));

      if (! fs::exists(fst_path)) {
        cerr << "error: unable to find " << fst_path.native_file_string() << "file\n";
        continue;
      }

      fs::path pattern_path = fs::change_extension(fst_path, ".pat");

      if (! fs::exists(pattern_path)
          || fs::last_write_time(fst_path) >= fs::last_write_time(pattern_path)) {
          // transform fst to XML

        cout << "compiling " << fst_path.native_file_string() << " to "
          << pattern_path.native_file_string() << endl;
        FST2_to_syntagm_pattern(fst_path, pattern_path, name, lingdef);
     
      } else {
        cout << "found an " << pattern_path.native_file_string() << endl;
      }

      assert(fs::exists(pattern_path));

      cerr << "adding " << pattern_path.native_file_string() << " to grammar ...\n";

      int idx = grm.add_pattern(pattern_path, name);
      
      cerr << "looking for subgraphs...\n";
      int nb = lookup_subpatterns(grm[idx], make_stack_pusher(stk));

      cout << nb << " more patterns added\n";
    }
  
    cout << "grammar is loaded (" << grm.size() << " patterns)." << endl;

    cout << "writting result to " << grammarpath.native_file_string() << endl;

    grm.write(grammarpath);
  
  } catch (exception & e) {
  
    cerr << "error: excepion caught : " << e.what() << endl;
    return 1;
  }

  cout << "done.\n";
}

