#ifndef _SYNTAGM_PATTERN_H_
#define _SYNTAGM_PATTERN_H_

#include <vector>

#include <boost/filesystem/path.hpp>

#include <outilex/lexical_mask.h>
#include <outilex/synt_mask.h>
#include <outilex/synt_output.h>


/* represent a syntagm pattern, 
 * obtained from a local grammar
 */


class syntagm_pattern {

public:

  typedef lexical_mask input_type;      // label in on lex transitions
  typedef synt_mask    synt_input_type; // label in on the synt transitions

  typedef synt_output  output_type;
  typedef synt_output  synt_output_type;
 
  struct transition {

    int to_;
    input_type  in_;
    output_type out_;

    inline transition() {}
    inline transition(xmlNodePtr node, ling_def * lingdef) { read_XML(node, lingdef); }
    inline transition(const input_type & in, const output_type & out, int to)
      : to_(to), in_(in), out_(out) { }

    inline int & to() { return to_; }
    inline int to() const { return to_; }

    inline input_type & in() { return in_; }
    inline const input_type & in() const { return in_; }

    inline output_type & out() { return out_; }
    inline const output_type & out() const { return out_; }

    static inline const xmlChar * xml_name() { return (const xmlChar *) "transition"; }
    void read_XML(xmlNodePtr node, ling_def * lingdef);
    void write_XML(xmlwriter & writer) const;
  };


  typedef std::vector<transition> transitions;
  typedef transitions::iterator trans_iterator;
  typedef transitions::const_iterator const_trans_iterator;

  struct synt_transition {

    int to_;
    synt_mask   in_;
    output_type out_; 

    inline synt_transition() : to_(-1), in_(NULL), out_() {}
    inline synt_transition(const synt_input_type & in, const synt_output_type & out, int to)
      : to_(to), in_(in), out_(out) { }

    inline synt_transition(xmlNodePtr node, ling_def * lingdef)
      : to_(-1), in_(NULL), out_() { read_XML(node, lingdef); }

    inline int to() const { return to_; }
    inline const synt_mask & in() const { return in_; }
    inline const output_type & out() const { return out_; }

    static inline const xmlChar * xml_name() { return (const xmlChar *) "synt_transition"; }
    void write_XML(xmlwriter & writer) const;
    void read_XML(xmlNodePtr node, ling_def * lingdef);
  };


  typedef std::vector<synt_transition> synt_transitions;
  typedef synt_transitions::iterator synt_trans_iterator;
  typedef synt_transitions::const_iterator const_synt_trans_iterator;


  struct state {

    bool final;
    transitions trans;
    synt_transitions synt_trans;

    state() : final(false), trans(), synt_trans() {}

    inline void clear() { final = false; trans.clear(); synt_trans.clear(); }


    inline void add_trans(const input_type & in, const output_type & out, int to) {
      trans.push_back(transition(in, out, to)); 
    }

    inline void add_synt_trans(const synt_input_type & in, const synt_output_type & out, int to) {
      synt_trans.push_back(synt_transition(in, out, to)); 
    }


    inline trans_iterator trans_begin() { return trans.begin(); }
    inline const_trans_iterator trans_begin() const { return trans.begin(); }

    inline trans_iterator trans_end() { return trans.end(); }
    inline const_trans_iterator trans_end() const { return trans.end(); }

    inline synt_trans_iterator synt_trans_begin() { return synt_trans.begin(); }
    inline const_synt_trans_iterator synt_trans_begin() const { return synt_trans.begin(); }

    inline synt_trans_iterator synt_trans_end() { return synt_trans.end(); }
    inline const_synt_trans_iterator synt_trans_end() const { return synt_trans.end(); }

    void write_XML(xmlwriter & writer) const;
    void read_XML(xmlNodePtr node, ling_def * lingdef);
  };

  inline syntagm_pattern(ling_def * ldef) : name(), lingdef(ldef), states() {}

  inline syntagm_pattern(const std::string & name_, ling_def * ldef)
    : name(name_), lingdef(ldef), states() {}

  inline syntagm_pattern(xmlNodePtr node, ling_def * lingdef) : states() { read_XML(node, lingdef); }
 
  inline syntagm_pattern(const boost::filesystem::path & path, ling_def * ldef) {
    read(path, ldef);
  }
  
  inline void set_name(const std::string & name_) { name = name_; }

  inline int start() const { return 0; }

  inline int size() const { return states.size(); }
  template<typename Pattern>
  inline void clear(const Pattern & pat) {
    clear();
    name = pat.get_name();
    lingdef = pat.get_lingdef(); 
  }

  inline void clear() { states.clear(); }

  inline void resize(int size) { states.resize(size); }

  inline int add_state() {
    int res = states.size();
    states.resize(res + 1);
    return res;
  }
  
  void add_trans(int from, const input_type & in, const output_type & out, int to) {
    states[from].add_trans(in, out, to);
  }

  void add_synt_trans(int from, const synt_input_type & in, const synt_output_type & out, int to) {
    states[from].add_synt_trans(in, out, to);
  }

  inline const std::string & get_name() const { return name; }
  inline ling_def * get_lingdef() const { return lingdef; }

  //inline bool & final(int q) { return states[q].final; }
  inline void set_final(int q, bool final = true) { states[q].final = final; }
  inline bool final(int q) const { return states[q].final; }

  inline trans_iterator trans_begin(int q) { return states[q].trans_begin(); }
  inline const_trans_iterator trans_begin(int q) const { return states[q].trans_begin(); }

  inline trans_iterator trans_end(int q) { return states[q].trans_end(); }
  inline const_trans_iterator trans_end(int q) const { return states[q].trans_end(); }

  inline synt_trans_iterator synt_trans_begin(int q) { return states[q].synt_trans_begin(); }
  inline const_synt_trans_iterator synt_trans_begin(int q) const {
    return states[q].synt_trans_begin();
  }

  inline synt_trans_iterator synt_trans_end(int q) { return states[q].synt_trans_end(); }
  inline const_synt_trans_iterator synt_trans_end(int q) const {
    return states[q].synt_trans_end();
  }

  static inline const xmlChar * xml_name() { return (const xmlChar *) "syntagm_pattern"; }
  void read_XML(xmlNodePtr node, ling_def * lingdef);
  void write_XML(xmlwriter & writer) const;
  void write_XML(const boost::filesystem::path & opath) const;

  void read(const boost::filesystem::path & path, ling_def * lingdef);

protected:
  std::string name;
  ling_def * lingdef;
  std::vector<state> states;
};

#endif
