#include <iostream>
#include <stack>

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/convenience.hpp>
                       
#include <outilex/FST2_to_syntagm_pattern.h>
#include <outilex/FST2_to_string_transducer.h>
#include <outilex/string_transducer.h>

#include <outilex/epsilon_removal.h>
#include <outilex/lingdef.h>
#include <outilex/grammar.h>

using namespace std;

namespace fs = boost::filesystem;


char * progname;


void usage(ostream & os = cerr, int ret = 1) {
  os << "usage: " << progname << " -l <lingdef> -r [ -synt | -trans | -lex ] [<fst2> ...]\n";
  exit(ret);
}

struct is_epsilon {
  pos_def * pos_epsilon;
  
  is_epsilon(pos_def * pos) : pos_epsilon(pos) {}
  bool operator()(const lexical_mask & m) const { return m.get_pos() == pos_epsilon; }
};


int main(int argc, char ** argv) try {

  fs::path lingdefpath;
  bool remove_epsilon = false;


  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }


  enum { UNSPEC = 0, STRING_FST, SYNT_FST } fst_type = UNSPEC;


  while (argc) {

    string arg = *argv;

    if (arg == "-h") {

      usage(cout, 0);

    } else if (arg == "-r") {

      remove_epsilon = true;

    } else if (arg == "-l") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);

    } else if (arg == "-synt") {

      fst_type = SYNT_FST; 

    } else if (arg == "-trans") {

      fst_type = STRING_FST; 

    } else if (arg == "-lex") {

      //fst_type = LEX_FST; 
      throw runtime_error("-lex not implemented\n");

    } else { // assume fst name

      if (lingdefpath.empty()) {
        cerr << "error: no lingdef specified\n";
        exit(1);
      }

      if (fst_type == UNSPEC) {
        cerr << "error: no FST type specified\n";
        exit(1);
      }


      ling_def lingdef(lingdefpath);
      fs::path fstpath(*argv, fs::native);
      if (! fs::exists(fstpath)) {
        cerr << "error: unable to find " << fstpath.native_file_string() << " file\n";
        exit(1);
      }

      fs::path outpath;

      switch (fst_type) {
      case STRING_FST: {

        outpath = fs::change_extension(fstpath, ".sfst");

        cout << "compiling " << fstpath.string() << " into " << outpath.string() << "...\n";


        string_transducer T1(& lingdef), T2(& lingdef);

        FST2_to_string_transducer(fstpath, T1);

        T1.check_for_final_outputs();
        if (remove_epsilon) {
          cout << "epsilon removal ...\n";
          fst_remove_epsilon(T1, T2, is_epsilon(lingdef.epsilon_pos()));
          T2.write(outpath);
        } else {
          T1.write(outpath);
        }
        cout << "done.\n";
      }
        break;

      case SYNT_FST: {

        outpath = fs::change_extension(fstpath, ".pat");
        cout << "compiling " << fstpath.native_file_string()
          << " into " << outpath.native_file_string() << "...\n";

        syntagm_pattern T1(& lingdef), T2(& lingdef);

        FST2_to_syntagm_pattern(fstpath, T1);

        if (remove_epsilon) {
          cout << "epsilon removal ...\n";
          synt_fst_remove_epsilon(T1, T2, is_epsilon(lingdef.epsilon_pos()));
          T2.write_XML(outpath);
        } else {
          T1.write_XML(outpath);
        }
        cout << "done.\n";
      }
        break;

      default:
        throw runtime_error("bad FST type?");
      }
    }
    argv++, argc--;
  }

  return 0;
} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}



