#include <iostream>
#include <string>
#include <vector>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/lingdef.h>
#include <outilex/rtn_grammar.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;

namespace {

template<typename OutputIterator>
int lookup_subpatterns(const rtn_pattern & fst, OutputIterator out) {

  int nbtrans = 0;
  for (int q = 0; q < fst.size(); ++q) {
    
    for (rtn_pattern::transitions::const_iterator tr = fst.trans_begin(q);
         tr != fst.trans_end(q); ++tr) {
      
      if (tr->in().type == rtn_input_type::SUBCALL) {
        const string & name = boost::get<string>(tr->in().v);
        //cout << name << " " << flush;
        *out = name;
        ++out;
      }
      ++nbtrans;
    }
  }
  return nbtrans;
}

void cleanup_pattern(rtn_pattern & pat, vector<string> & badnames) {

  for (int q = 0; q < pat.size(); ++q) {

    for (rtn_pattern::transitions::iterator tr = pat.trans_begin(q);
         tr != pat.trans_end(q);) {
      
      if (tr->in().type == rtn_input_type::SUBCALL
          && find(badnames.begin(), badnames.end(), boost::get<string>(tr->in().v))
             != badnames.end()) {

        tr = pat.trans(q).erase(tr);
      } else {
        ++tr;
      }
    }
  }
}

void cleanup_grammar(rtn_grammar & gram, vector<string> & badnames) {

  for (int i = 0; i < gram.size(); ++i) {
    cleanup_pattern(gram[i], badnames);
  }
}


char * progname;

void usage() {
  cout << "usage: " << progname << " -l <lingdef> <fstpattern>\n";
  exit(0);
}

} // namespace anonymous

int main(int argc, char ** argv) try {

  fs::path fstpath, lingdefpath;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }

  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {
  
    string arg = *argv;

    if (arg == "-l") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);

    } else if (arg == "-h") {
    
      usage();
    
    } else {
      fstpath = fs::path(arg, fs::native);
    }
    argv++, argc--;
  }

  if (fstpath.empty() || lingdefpath.empty()) { cerr << "bad args\n"; exit(1); }

  ling_def * lingdef = new ling_def(lingdefpath); 

  rtn_grammar gram(lingdef);

  fs::path grampath = fs::change_extension(fstpath, ".rtn");


  fs::path dir(fstpath.branch_path());

  string maingraph = fs::basename(fstpath);

  vector<string> stack;
  vector<string> badnames;
  stack.push_back(maingraph);

  int totalstates = 0, totaltrans = 0;

  while (! stack.empty()) {

    string name = stack.back();
    stack.pop_back();
  
    if (gram.get_pat_idx(name) != -1) { continue; } // already proceeded

    cout << "processing " << name << " ... " << flush;
  
    fstpath = dir / (name + ".rpat");
  
    if (! fs::exists(fstpath)) {
      cerr << "error: unable to find " << fstpath.string() << endl;
      badnames.push_back(name);
      continue;
    }

    int idx = gram.add_pattern(fstpath, name);
    int nbtrans = lookup_subpatterns(gram[idx], back_inserter(stack));

    cout << '(' << gram[idx].size() << " states, " << nbtrans << " transitions)\n";

    totalstates += gram[idx].size();
    totaltrans  += nbtrans;
  }

  if (! badnames.empty()) {
    cout << "grammar cleanup ...\n";
    cleanup_grammar(gram, badnames);
  }
  cout << "writing grammar into " << grampath.string() << '\n'
    << gram.size() << " patterns\n"
    << "total of " << totalstates << " states and " << totaltrans << " transitions.\n";

  gram.write(grampath);
  cout << "done.\n";

  return 0;

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

