#ifndef _LEXICAL_MASK2_H_
#define _LEXICAL_MASK2_H_

#include <outilex/form_set.h>
#include <outilex/feat_set.h>

class ling_def;
class lexical_mask;


/* a lexical mask describe the set of lexical entries which match with
 * the set of specified features
 */

class lexical_mask2 {

public: // make all data members public

  mutable form_set form;
  mutable form_set lemma;
  mutable const pos_def * pos;
  // -1 means unspecified, 0 means that this mask is void, i.e. it descibes empty set

  typedef std::map<attr_def *, feat_set> feats_map;
  mutable feats_map feats;

private:

  /* used internally to clear a mask which is know to describe the empty set.
   */

  void clear_me() const {
    form = 0; lemma = 0; 
    feats.clear();
    pos = 0;
  }

public:

  /* default constructor, matches with the whole lexic
   */
  lexical_mask2(pos_def * pos_ = pos_unspec) : form(), lemma(), pos(pos_), feats() {}

  /* construct from an xml description */
  lexical_mask2(xmlNodePtr node, ling_def * lingdef)
    : form(), lemma(), pos(pos_unspec), feats() { read_xml(node, lingdef); }

  /* construct from a text description */
  lexical_mask2(const std::string & txt, ling_def * lingdef)
    : form(), lemma(), pos(pos_unspec), feats() { read_text(txt, lingdef); }

  lexical_mask2(const lexical_mask & m);


  const pos_def * get_pos() const { return pos; }

  /* return true if mask describe nothing (empty set)
   * false otherwise
   */

  bool empty() const { return pos == 0; }
  
  /* clear mask, and make it describe empty set */
  
  void clear() { form.clear(); lemma.clear(); feats.clear(); pos = 0; }

  /* make the mask describe all the lexic */

  void unspec() { form.unspec(); lemma.unspec(); feats.clear(); pos = pos_unspec; }
  
  /* int assignement operator
   * m = 0 -> mask is void (match with nothing)
   * m = 1 (!= 0) -> mask match everything
   */

  lexical_mask2 & operator=(int v) {
    if (v == 0) {
      clear();
    } else { unspec(); }
    return * this;
  }

  /* int cast operator
   * return 1 if mask match with something (theoretically)
   *        0 otherwise
   */

  operator const void*() const { return empty() ? 0 : this; }

  bool operator!() const { return empty(); }


  /* ensemblist operations */

  /* return true if *this is included in m
   */
  bool in(const lexical_mask2 & m) const;


  /* return true if the two masks are not disjoint */

  bool intersects(const lexical_mask2 & b) const;

  bool intersects(const lexical_mask & b) const;


  static bool intersects(const lexical_mask2 & a, const lexical_mask2 & b) {
    return a.intersects(b);
  }

  static bool intersects(const lexical_mask2 & a, const lexical_mask & b) {
    return a.intersects(b);
  }

  static bool intersects(const lexical_mask & a, const lexical_mask2 & b) {
    return b.intersects(a);
  }



  bool operator==(const lexical_mask2 & b) const;

  bool operator!=(const lexical_mask2 & b) const { return ! (*this == b); }

  /* assign to *this the intersection of a and b
   * return true if result is not empty
   */

  bool set_inter(const lexical_mask2 & a, const lexical_mask2 & b);
  
  /* assign to *this the intersection of *this and b */
  void set_inter(const lexical_mask2 & b);
  void operator&=(const lexical_mask2 & b) { set_inter(b); }


  template<class OutputIterator>
  static void minus(const lexical_mask2 & a, const lexical_mask2 & b, OutputIterator out) {

    // std::cerr << "lexmask: " << a << " minus " << b << std::endl;

    if (! b.in(a)) { // first, make sure that b is included in a
      lexical_mask2 bb;
      bb.set_inter(a, b);
      minus(a, bb, out);
      return;
    }

    if (a.empty()) { // a is empty, return empty set
      return;
    }

    if (b.empty()) { // b is empty, return a
      *out = a; ++out;
      return;
    }

    lexical_mask2 m(a);


    if (b.pos != pos_unspec) {

      if (m.pos == pos_unspec) {

        std::vector<const pos_def *> poss;
        ::minus(m.pos, b.pos, back_inserter(poss));

        for (std::vector<const pos_def *>::const_iterator it = poss.begin();
             it != poss.end(); ++it) {
          m.pos = *it;
          *out = m; ++out;
        }

        m.pos = b.pos;
      }

      for (feats_map::const_iterator itb = b.feats.begin(); itb != b.feats.end(); ++itb) {

        attr_def * attr  = (*itb).first;
        const feat_set & featb = (*itb).second;

        feats_map::iterator itm = m.feats.find(attr);

        if (itm == m.feats.end()) {  // add unspec featset if attribute not specified in a
          itm = m.feats.insert(m.feats.begin(), std::make_pair(attr, feat_set(attr->get_type())));
        }

        feat_set & featm = (*itm).second;
        featm -= featb;

        if (featm) {
          *out = m; ++out;
        }
        featm = featb;
      }
    } else { assert(m.pos == pos_unspec); }

    // compute form at the end

    m.form -= b.form;
    if (m.form) { // b form is a strict subset of a form
      *out = m; ++out;
    }
    m.form = b.form;

    m.lemma -= b.lemma;
    if (m.lemma) {
      *out = m; ++out;
    }
    //m.lemma = b.lemma;
  }



  /* serialisation functions */

  //static const xmlChar * xml_name() { return (const xmlChar *) "lexical_mask2"; }
  void read_xml(xmlNodePtr node, ling_def * lingdef);
  void write_xml(xmlwriter & writer) const;


  void read_text(const std::string & txt, ling_def * lingdef);
  void dump_text(std::ostream & os) const;
};



inline std::ostream & operator<<(std::ostream & os, const lexical_mask2 & m) {
  m.dump_text(os); return os;
}

#endif

