#include <iostream>
#include <vector>

#include <cassert>

#include <outilex/xml.h>
#include <outilex/xml-names.h>
#include <outilex/form_set.h>
#include <outilex/disjoint.h>
#include <outilex/stringtok.h>
#include <outilex/escape.h>


using namespace std;
using namespace xmlnames;

/* return true if a is a subset of b */

bool form_set::in(const form_set & a, const form_set & b) {

  if (a.neg) {

    //2 negations, retourne vrai si toutes les formes niees dans b sont egalement niees dans a
    if (b.neg) {

      return std::includes(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end());

    } else { // a is negative (non borne) and b positive (borne), return false

      return false;
    }

  } else { // a is positive

    if (b.neg) { // and b is negative, check that all form specified in a are not negated in b

      return disjoint(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end()); 

    } else { // both are positive, return true if all form specified in a are also present in b

      return std::includes(b.vals.begin(), b.vals.end(), a.vals.begin(), a.vals.end());
    }
  }

  assert("never reached!!!" && 0);
  return false;
}


bool form_set::intersects(const form_set & b) const {

  if (neg) {
    
    /* si les deux sont negatif, l'intersection est necessairement non vide */

    if (b.neg) { return true; }
    
    /* a negatif et b positif 
     * l'intersection est non vide si il existe une forme dans b qui n'est pas
     * niee dans a
     */
    return ! std::includes(vals.begin(), vals.end(), b.vals.begin(), b.vals.end());
  
  } else {

    if (b.neg) { // symetric
      return ! std::includes(b.vals.begin(), b.vals.end(), vals.begin(), vals.end());
    }
    
    /* a et b positifs :
     * l'intersection est non vide ssi il existe une forme dans a presente dans b
     */
    return ! disjoint(vals.begin(), vals.end(), b.vals.begin(), b.vals.end());
  }
}


bool form_set::set_inter(const form_set & a, const form_set & b) {

  if (& a == this) { form_set aa = a; return set_inter(aa, b); }
  if (& b == this) { form_set bb = b; return set_inter(a, bb); }

  if (! a.neg) {

    if (! b.neg) { // a and b are positive, compute normal intersection

      neg = false;
      vals.clear();
      set_intersection(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end(),
                       back_inserter(vals));

    } else { // a is pos and b is neg, make result with all string in a which are not present in b

      neg = false;
      vals.clear();
      set_difference(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end(),
                     back_inserter(vals));
    }

  } else { // a is negative

    if (! b.neg) { // b is pos, compute string in b which are not in a

      neg = false;
      vals.clear();
      set_difference(b.vals.begin(), b.vals.end(), a.vals.begin(), a.vals.end(),
                     back_inserter(vals));

    } else { // a and b are negative, make an union

      neg = true;
      vals.clear();
      set_union(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end(),
                back_inserter(vals));
    }
  }

  return ! empty();
}


void form_set::set_inter(const form_set & b) {

  list<string> nvals;

  if (! neg) {

    if (! b.neg) { // a and b are positive, compute normal intersection

      std::set_intersection(vals.begin(), vals.end(), b.vals.begin(), b.vals.end(),
                            back_inserter(nvals));

    } else { // a is pos and b is neg, make result with all string in a which are not present in b

      std::set_difference(vals.begin(), vals.end(), b.vals.begin(), b.vals.end(),
                          back_inserter(nvals));
    }

  } else { // a is negative

    if (! b.neg) { // b is pos, compute string in b which are not in a

      neg = false;
      std::set_difference(b.vals.begin(), b.vals.end(), vals.begin(), vals.end(),
                          back_inserter(nvals));

    } else { // a and b are negative, make an union

      neg = true;
      std::set_union(vals.begin(), vals.end(), b.vals.begin(), b.vals.end(),
                     back_inserter(nvals));
    }
  }

  vals.swap(nvals);
}


bool form_set::set_minus(const form_set & a, const form_set & b) {

  if (& a == this) { form_set aa = a; return set_minus(aa, b); }
  if (& b == this) { form_set bb = b; return set_minus(a, bb); }

  vals.clear();

  if (a.neg) {
  
    if (b.neg) { // !pomme!poire \ !pomme!orange = !pomme!poire inter pomme,orange = orange

      neg = false;
      
      set_difference(b.vals.begin(), b.vals.end(), a.vals.begin(), a.vals.end(),
                     back_inserter(vals));
    } else { //!pomme!poire \ pomme,orange = !pomme!poire!orange
      
      neg = true;
      set_union(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end(),
                back_inserter(vals));
    }
  } else {
  
    if (b.neg) { // pomme,poire \ !pomme!orange = pomme,poire inter pomme,orange = pomme

      neg = false;
      set_intersection(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end(),
                       back_inserter(vals));
    
    } else { // pomme,poire \ pomme,orange = poire

      neg = false;
      set_difference(a.vals.begin(), a.vals.end(), b.vals.begin(), b.vals.end(),
                     back_inserter(vals));
    }
  }

  return ! empty();
}


void form_set::set_minus(const form_set & b) {

  list<string> nvals;

  if (neg) {
  
    if (b.neg) { // !pomme!poire \ !pomme!orange = !pomme!poire inter pomme,orange = orange

      neg = false;
      set_difference(b.vals.begin(), b.vals.end(), vals.begin(), vals.end(),
                     back_inserter(nvals));

    } else { //!pomme!poire \ pomme,orange = !pomme!poire!orange
      
      neg = true;
      set_union(vals.begin(), vals.end(), b.vals.begin(), b.vals.end(),
                back_inserter(nvals));
    }

  } else {
  
    if (b.neg) { // pomme,poire \ !pomme!orange = pomme,poire inter pomme,orange = pomme

      neg = false;
      set_intersection(vals.begin(), vals.end(), b.vals.begin(), b.vals.end(),
                       back_inserter(nvals));
    
    } else { // pomme,poire \ pomme,orange = poire

      neg = false;
      set_difference(vals.begin(), vals.end(), b.vals.begin(), b.vals.end(),
                     back_inserter(nvals));
    }
  }

  vals.swap(nvals);
}



/* serialisation */

void form_set::read_text(const std::string & text) {

  if (text.empty()) { // empty means unspec ... 
    unspec(); return;
  }

  if (text[0] == '!') {
    // '!' only means void (in contradiction with the obvious form... but more intuitive)
    if (text.size() == 1) {
      clear(); return;
    }
    neg = true;
  } else {
    neg = false;
  }

  vals.clear();
  stringtok_unescaped(text, "!+", back_inserter(vals));
  vals.sort();
}


namespace {
const char * SPECIAL_CHARS = "!+,.";
};

void form_set::dump_text(ostream & os) const {

  if (neg) {
    if (vals.empty()) { // unset
      return;
    }
    os << '!';
  } else {
    if (vals.empty()) { // void
      os << '!'; return;
    }
  }

  list<string>::const_iterator it = vals.begin();
  os << escape(*it, SPECIAL_CHARS);

  while (++it != vals.end()) {
    os << '+' << escape(*it, SPECIAL_CHARS);
  }
}


void form_set::dump_unescaped(ostream & os) const {

  if (neg) {
    if (vals.empty()) { // unset
      return;
    }
    os << '!';
  } else {
    if (vals.empty()) { // void
      os << '!'; return;
    }
  }

  list<string>::const_iterator it = vals.begin(), end = vals.end();
  os << *it;

  while (++it != end) {
    os << '+' << *it;
  }
}


void form_set::read_xml(xmlNode * node) {

  clear();

  if (node == NULL) { return; }

  char * text = xmlGetProp(node, NEG_ATTR);

  if (text && text[0] == '1') {
    neg = true;
  } else {
    neg = false;
  }
  xmlFree(text);

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, FEAT_ELEM) == 0) {
      text = (char *) xmlNodeGetContent(node);
      vals.push_back(text);
      xmlFree(text);
    }
    node = node->next;
  }
  vals.sort();
}


void form_set::write_xml(xmlwriter & writer, const string & name) const {

  writer.start_element(name);

  if (neg) {
    writer.write_attribute(NEG_ATTR, "1");
  }

  if (! vals.empty()) {
  
    for (list<string>::const_iterator it = vals.begin(); it != vals.end(); ++it) {
     writer.write_element(FEAT_ELEM, *it);
    }
  }

  writer.end_element();
}

