#include <iostream>
#include <string>
#include <stdexcept>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/tokenizer.hpp>

#include <outilex/stringtok.h>
#include <outilex/lg_table.h>

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

typedef tokenizer<escaped_list_separator<char> > csv_tokenizer;

lg_table::param_type lg_table::get_parameter_type(const string & param) {

  if (param.empty()) { return invalid_param; }

  switch (param[0]) {
  case '!':
    return neg_param;
  case '$':
    return prefix_param;
  case '%':
    return rowid_param;
  case ':':
    return default_param;
  default:
    return normal_param;
  }
}


bool lg_table::resolve_prefix_parameter(int e, const string & param, string & res) const {

  static bool warning = false;

  if (! warning) {
    cerr << "warning: prefix parameter is badly implemented\n";
    warning = true;
  }

  bool found = false;
  int ln = param.size();

  map<string, int>::const_iterator it = properties.begin();

  
  while (it != properties.end()) {
    
    if (it->first.substr(0, ln) == param) { // we find something
      found = true;
      if (resolve_elementary_parameter(e, it->first, default_param, res)) {
        return true;
      }
      // value is '-' for this prop, go on with search
    }
    ++it;
  }

  if (not found) {
    cerr << "warning prefix '" << param << "' not found\n";
    return false;
  }
  return true;
}

bool lg_table::resolve_elementary_parameter(int e, const string & param,
                                            param_type type, string & res) const {

  map<string, int>::const_iterator it = properties.find(param);

  if (it == properties.end()) {
    if (not_found.find(param) == not_found.end()) {
      not_found.insert(param);
      cerr << "error: property '" << param << "' not found\n";
    }
    return false;
  }

  int prop = it->second;
  const string & val = table[e][prop];

  if (type == neg_param) {
    if (val == "-" || val == "/") { res = "<E>"; return true; } 
    return false;
  }

  if (type == default_param) {
    if (val != "-" && val != "/") { res = defaults[prop]; return true; }
    return false;
  }

  assert(type == normal_param);

  if (val == "-" || val == "/") { return false; }
  if (val == "+" || val == "*" || val == "~") { res = "<E>"; return true; }

  res = val;
  return true;
}


bool lg_table::resolve(int e, const string & param, std::string & res) const {

  param_type type = get_parameter_type(param);

  if (type == invalid_param) {
    cerr << "invalid parameter : " << param << endl;
    return false; 
  }

  if (type == rowid_param) { res = lexical_cast<string>(e); return true; }

  if (type == prefix_param) {
    return resolve_prefix_parameter(e, param.substr(1), res);
  }

  if (type != normal_param) {
    return resolve_elementary_parameter(e, param.substr(1), type, res);
  } 

  return resolve_elementary_parameter(e, param, normal_param, res);
}


void lg_table::load(const fs::path & path, char fieldsep) {

  ncol = 0;
  properties.clear();
  defaults.clear();
  table.clear();

  fs::ifstream is(path);
  if (! is) { throw runtime_error("lg_table load: file not found : " + path.string()); }
  
  string line;


  /* get headers */

  getline(is, line);
  
  /*
  vector<string> split;
  stringtok(line, fieldsep, back_inserter(split));

  ncol = split.size();

  for (int i = 0; i < ncol; ++i) {
    properties[split[i]] = i;
  }
  */


  csv_tokenizer tok(line);
  ncol = 0;
  for (csv_tokenizer::iterator it = tok.begin(); it != tok.end(); ++it) {
    properties[*it] = ncol;
    ncol++;
  }

  /* get "default value" row */

  getline(is, line);
  
  tok.assign(line);

  defaults.reserve(ncol);
  for (csv_tokenizer::iterator it = tok.begin(); it != tok.end(); ++it) {
    defaults.push_back(*it);
  }
  if (defaults.size() != ncol) {
    cerr << "error: bad header line : " << line << endl;
    throw runtime_error("lg_table::load: bad table format");
  }

  /*
  split.clear();
  stringtok(line, fieldsep, back_inserter(split));

  if (split.size() != ncol) {
    cerr << "error: lg_table: bad line : " << line << endl;
    throw runtime_error("lg_table::load: bad table format");
  }
  
  defaults.resize(ncol);
  for (int i = 0; i < ncol; ++i) {
    defaults[i] = split[i];
  }
  */

  /* read entries */
  int e = 0;
  while (getline(is, line)) {
  
    tok.assign(line);
 
    table.push_back(vector<string>());

    vector<string> & row = table[e];
    row.reserve(ncol);

    for (csv_tokenizer::iterator it = tok.begin(); it != tok.end(); ++it) {
      row.push_back(*it);
    }

    if (row.size() != ncol) {
      cerr << "lg_table: bad entry line: " << line << endl;
      throw runtime_error("lg_table: bad table format");
    }
    e++;
  }
  //cerr << "table loaded (" << ncol << " properties and " << e << " entries)\n";
}


