#ifndef _DICO_H_
#define _DICO_H_

#include <iosfwd>

#include <vector>
#include <string>
#include <string>

#include <boost/filesystem/path.hpp>

#include <outilex/dic_fsa2.h>
#include <outilex/dic_entry_value.h>
#include <outilex/dic_lex_entry.h>
#include <outilex/unicode.h>


static const int DICOMAGIC = 0x0D1C0F5A;

/**
 * @brief Class describing a dictionary represented as an automaton
*/
class dico {

public:
  typedef dic_fsa2<dic_entry_value, UChar, std::vector<std::set<dic_entry_value> > > FSA;

  FSA fsa;
  std::vector<std::string> entries_feats;
  std::vector<std::string> inflex_feats;

  int priority;

  void read_state(std::istream & is, int no);
  void read_values(std::istream & is); 
  void read_feats(std::istream & is);
  void read_fsa(std::istream & is);



public:

  dico(int prio = 10)
    : fsa(), entries_feats(), inflex_feats(), priority(prio) {}

  dico(const boost::filesystem::path & filename, int prio = 10)
    : priority(prio) { read(filename); }

  void read(const boost::filesystem::path & path);

  struct position_;

  struct trans_iterator {
    
    const dico * dic;
    FSA::const_trans_iterator it;
    int idx;

    trans_iterator() : dic(NULL), it(), idx(-1) {}
    trans_iterator(const dico & d, const FSA::const_trans_iterator & i, int id)
      : dic(&d), it(i), idx(id) {}

    void operator++() { idx += dic->fsa[it->second].card; ++it; }
    bool operator!=(const trans_iterator & b) { return it != b.it; }

    UChar label() { return it->first; }
    position_ follow() { return position_(*dic, it->second, idx); }
  };

  struct position_ {
    
    const dico * dic;
    int qno;
    int idx;

    position_(const dico & d) : dic(&d), qno(0), idx(0) {}
    position_(const dico & d, int q, int i) : dic(&d), qno(q), idx(i) {}

    bool final() const { return dic->fsa.states[qno].final; }
    trans_iterator trans_begin() const { return trans_iterator(*dic, dic->fsa.states[qno].begin(),
                                                               idx + final()); }
    trans_iterator trans_end() const {
      return trans_iterator(*dic, dic->fsa.states[qno].end(), -1);
    }


    bool operator==(const position_ & b) const {
      return dic == b.dic && qno == b.qno && idx == b.idx;
    }

    bool operator!=(const position_ & b) const {
      return ! (*this == b);
    }
    bool operator<(const position_ & b) const {
      if (dic != b.dic) { return dic < b.dic; }
      if (qno != b.qno) { return qno < b.qno; }
      return idx < b.idx; 
    }
  };


  // we need the form path to compute to lemma ...

  struct position {

    std::vector<UChar> form;
    position_ pos;

    explicit position(const std::vector<UChar> & f, const position_ & p) : form(f), pos(p) {}
    explicit position(const dico & dico) : form(), pos(dico) {}

    int priority() const { return pos.dic->priority; }

    bool operator<(const position & b) const {
      /* sort by priority first */
      if (priority() != b.priority()) { return priority() > b.priority(); }
      if (pos != b.pos) { return pos < b.pos; }
      return form < b.form;
    }
  };

  void make_lex_entry(dic_lex_entry & e, const std::vector<UChar> & form,
                      const dic_entry_value & v) const;

  template<typename OutputIterator>
  int make_lex_entries(const std::vector<UChar> & form, int idx, OutputIterator out) const {

    int res = 0;
    //std::cerr << "entering make_lex_entries\n";
    dic_lex_entry e;

    /* back to utf8 canonical form */

    UErrorCode uerror = U_ZERO_ERROR;
    std::vector<UChar> buf(form.size());
    int lenb = unicode::normalize(buf, UNORM_NFC, & form[0], form.size(), uerror);

    e.form = unicode::utf8_from_u_str(& buf[0], lenb, uerror);

    //std::cerr << "unicode stuffs done, form = " << e.form << "\n";

    const std::set<dic_entry_value> & vals = fsa.values[idx];
    //std::cerr << vals.size() << " matches\n";

    for (std::set<dic_entry_value>::const_iterator it = vals.begin(); it != vals.end(); ++it) {
      e.feats.clear(); e.lemma.clear(); e.POS.clear();
      make_lex_entry(e, form, *it); 
      *out = e;
      ++out;
      ++res;
    }

    //std::cerr << "out of make_lex_entries\n";
    return res;
  }


  // debugging purpose


  friend struct dump_entries;
  void dump(std::ostream & os);

#if 0
  const FSA::state & states(int qno) { return fsa.states[qno]; }

  template<typename Iterator>
  int lookup_idx(Iterator begin, Iterator end) const {
    
    int idx = 0, qno = 0;

    while (begin != end) {

      const FSA::state & q = states(qno);

      if (q.final) { ++idx; }
 
      FSA::const_trans_iterator it;
      
      for (it = q.trans.begin(); it != q.trans.end() && (*it).first < *begin; ++it) { idx += states((*it).second).card; }

      if (it == q.trans.end() || (*it).first != *begin) { return -1; }

      qno = (*it).second;
      ++begin;
    }

    return idx;
  }

  template<typename Iterator>
  bool lookup(Iterator begin, Iterator end, std::set<dic_entry_value> & res) {
    int idx = lookup_idx(begin, end);
    if (idx == -1) { return false; }
    res = fsa.values[idx];
    return true;
  }

  template<typename Container>
  bool lookup(Container & form, std::set<dic_entry_value> & res) { return lookup(form.begin(), form.end(), res); }

#endif
};

#endif

