#ifndef _DIC_FSA2_H_
#define _DIC_FSA2_H_

#include <iostream>
#include <fstream>
#include <iterator>
#include <vector>
#include <map>
#include <set>
#include <string>
#include <stack>
#include <cassert>
#include <algorithm>
#include <functional>

#include <outilex/olivtree2.h>




template<typename Value = std::string, typename Character = char,
         typename ValueTable = oliv_tree2<std::set<Value> > >
class dic_fsa2 {

public:

  typedef Value value_t;
  typedef Character char_t;
  typedef ValueTable value_table_t;


  typedef std::map<char_t, int> transitions_t;
  typedef std::pair<char_t, int> transition_t;

  typedef typename transitions_t::iterator trans_iterator;
  typedef typename transitions_t::const_iterator const_trans_iterator;


  class state {

  public:

    typedef typename transitions_t::iterator trans_iterator;
    typedef typename transitions_t::const_iterator const_trans_iterator;

    bool final;
    int nb_in;              // nb of in-transitions
    int card;
 
    transitions_t trans; // transitions


    state() : final(false), nb_in(0), card(0), trans() {}

    trans_iterator find_trans(char_t c) { return trans.find(c); }
    const_trans_iterator find_trans(char_t c) const { return trans.find(c); }

    trans_iterator begin() { return trans.begin(); }
    const_trans_iterator begin() const { return trans.begin(); }

    trans_iterator end() { return trans.end(); }
    const_trans_iterator end() const { return trans.end(); }


    /* comparaison operators */
    
    friend bool operator<(const state & a, const state & b) {
  
      if (a.final != b.final) {
	return b.final;
      }
     
      if (a.card != b.card) { return a.card < b.card; }     

      return a.trans < b.trans;
    }
  

    friend bool operator==(const state & a, const state & b) {
  
      if ((a.final != b.final) || (a.trans != b.trans)) { return false; }

      assert(a.card == b.card);
      return true;
    }

    
    void swap(state & q) {
      std::swap(final, q.final);
      std::swap(nb_in, q.nb_in);
      std::swap(card, q.card);
      trans.swap(q.trans);
    }



    void dump(std::ostream & os = std::cerr) const {
    
      os << "[nbin:" << nb_in << "card:" << card << " F:" << final << "] : ";
      for (const_trans_iterator it = trans.begin(); it != trans.end(); ++it) {
        os << "(" << (*it).first << ", " << (*it).second << ") ";
      }
    }

    friend std::ostream & operator<<(std::ostream & os, const state & s) {
      s.dump(os);
      return os;
    }
 
    friend std::ostream & operator<<(std::ostream & os, const transition_t & t) {
      os << "(" << t.first << "," << t.second << ")\n";
      return os;
    }

    friend std::ostream & operator<<(std::ostream & os, const transitions_t & trans) {
      os << "[";
      std::copy(trans.begin(), trans.end(), std::ostream_iterator<transition_t>(os, ","));
      os << " ]";
      return os;
    }
  };



  std::vector<state>  states;

  value_table_t values;

  dic_fsa2(bool add_initial = true) : states(), values() {
    if (add_initial) {
      states.push_back(state()); // add initial state
    }
  }

  void clear(bool add_initial = true) {
    states.clear();
    values.clear();
    if (add_initial) { states.push_back(state()); }
  }


  /* lookup */


  template<typename char_const_iterator>
  int lookup_idx(char_const_iterator beg, const char_const_iterator & end, int q = 0) const {
  
    if (size() == 0) { std::cerr << "warning: lookup_idx: fsa automaton is empty\n"; return -1; }

    int idx = 0;
    typename state::const_trans_iterator it;

    while (beg != end) {

      if (states[q].final) { idx++; }

      for (it = states[q].trans.begin(); it != states[q].trans.end() && (*it).first < *beg; ++it) {
        idx += states[(*it).second].card;
      }
      if ((it == states[q].trans.end()) || (*it).first != *beg) { return -1; }
      
      q = (*it).second;
      ++beg;
    }

    // beg == end

    if (states[q].final) { return idx; }
    return -1;
  }


  template<class string_t>
  inline int lookup_idx(const string_t & form) const {
    return lookup_idx(form.begin(), form.end());
  }


  template<class char_const_iterator>
  bool lookup(char_const_iterator begin, const char_const_iterator & end,
              std::set<value_t> & value) const {

    int idx = lookup_idx(begin, end);
    if (idx == -1) { value.clear(); return false; }
    value = values[idx];
    return true;
  }

  template<class string_t>
  inline bool lookup(const string_t & form, std::set<value_t> & value) const {
    return lookup(form.begin(), form.end(), value);
  }


  inline int size() const { return states.size(); }

  state & operator[] (int i) { return states[i]; }
  const state & operator[] (int i) const { return states[i]; }
  
  template<class char_const_iterator>
  int add_if_not_here(char_const_iterator beg, const char_const_iterator & end, int q = 0) {

    int idx;
    if ((idx = lookup_idx(beg, end, q)) != -1) { // already here
      //cerr << "add_if_not_here: already here " << idx << '\n';
      return idx;
    }
 
    /* new form */

    /* find common prefix of form and of the internal lexico tree of the automaton
     * (until nb_in > 1)
     */
 
    typename state::trans_iterator it;
    idx = 0;

    while (beg != end) {

      states[q].card++; // update cardinal during travel

      if (states[q].final) { idx++; }
      
      for (it = states[q].trans.begin();
           it != states[q].trans.end() && (*it).first < (*beg); ++it) {

        idx += states[(*it).second].card;
      }

      if ((it == states[q].trans.end()) || (*it).first != (*beg)) { /* create a new path */

        int newq = new_state();
        states[newq].nb_in = 1;

        states[q].trans[*beg] = newq;

        q = newq; ++beg;

      } else if (states[(*it).second].nb_in > 1) { /* dup the state */

        state & q1 = states[(*it).second];

        q1.nb_in--;

        // this don't work becaus dup_state invlidate it
        // q = states[q].trans[*c] = dup_state(q1);
        // q = ((*it).second = dup_state(q1));

        int newq = dup_state(q1);
        states[newq].nb_in = 1;

	states[q].trans[*beg] = newq;

        q = newq; ++beg;

      } else { /* still in internal lexicotree */
        ++beg; q = (*it).second;
      }
    }

    assert(! states[q].final);

    states[q].final = true;
    states[q].card++;

    /* make room */

    values.insert(idx, std::set<value_t>());

    return idx;
  }


  template<class char_const_iterator>
  void add_entry(char_const_iterator beg, const char_const_iterator & end, const value_t & val) {

    //#warning "optimize here (insert after add...)"

    int idx = add_if_not_here(beg, end);
    //std::cerr << "add_if_not_here=" << idx << endl;
    // values.check_integrity();
    values[idx].insert(val);
  } 


  template<typename char_const_iterator, typename Container>
  void add_entry(char_const_iterator begin, const char_const_iterator & end,
                 const Container & vals) {
    int idx = add_if_not_here(begin, end);
    values[idx].insert(vals.begin(), vals.end());
  }

  template<class string_t>
  inline void add_entry(const string_t & form, const value_t & v) {
    add_entry(form.begin(), form.end(), v);
  }

  template<class string_t, typename Container>
  void add_entry(const string_t & form, const Container & vals) {
    add_entry(form.begin(), form.end(), vals);
  }


 
  class LessState {
  public:
    const dic_fsa2 & dic;
    LessState(dic_fsa2 & _dic) : dic(_dic) {}

    bool operator()(int q1, int q2) {
      return dic.states[q1] < dic.states[q2];
    }
  };




  /* MINIMISATION */


  void calc_heights(std::vector<std::vector<int> > & res) const {

    //  std::cerr << "calc_height()\n";

    std::vector<int> height(states.size(), -1);
    std::stack<int> stck;

    stck.push(0); // begin with initial state

    while (! stck.empty()) {

      int id = stck.top();

      //std::cerr << id << " ";

      if (height[id] != -1) { stck.pop(); continue; }

      const state & q = states[id];
      typename state::const_trans_iterator it;

      height[id] = 0;
      for (it = q.trans.begin(); (it != q.trans.end()) && (height[(*it).second] != -1); ++it) {
        height[id] = std::max(height[id], height[(*it).second] + 1);
      }

      if (it == q.trans.end()) { // we have height[i]

        stck.pop();

      } else {

        height[id] = -1;
        while (it != q.trans.end()) {
          if (height[(*it).second] == -1) { stck.push((*it).second); }
          ++it;
        }
      }
    }

    // std::cerr << "height[0] = " << height[0] << "\n";

    res.clear();
    res.resize(height[0] + 1); // initial state has maximal height

    //bool inaccessible = false;

    for (int i = 0; i < height.size(); i++) {
      
      // std::cerr << "height[" << i <<  "] = " << height[i] << "\n";

      if (height[i] == -1) {
        std::cerr << "ERROR: state " << i << " is inaccessible\n";
        //dump();
        //dump_dot("fst_dic.dot");
        assert(states[i].nb_in == 0);
        abort();
      }
 
      // assert((i == 0) || (height[i] < height[0])); 

      res[height[i]].push_back(i);
    }
  }


  void minimize() {

    /* vector of states indexed by their height */

    //std::cerr << "minimize() nbstates = " << states.size() << "\n";
   
    std::vector<std::vector<int> > heights;

    calc_heights(heights);


    /* table of representants */
    
    std::vector<int> repr(states.size());

    for (int i = 0; i < states.size(); i++) { /* begin with each state being its own representant */
      repr[i] = i; 
    }


    for (int h = 0; h < heights.size(); h++) {

      /* remap transitions to point to their equiv state representant
       */

      for (int i = 0; i < heights[h].size(); i++) { // for each state in height[l]
        
        state & q = states[heights[h][i]];
        typename state::trans_iterator it;
        
        for (it = q.trans.begin(); it != q.trans.end(); ++it) { // for each trans
          
          (*it).second = repr[(*it).second];   // repoint trans to equiv state
          //    states[(*it).second].nb_in++;        // and DO NOT update its in-trans num !
        }
      }


      /* we can now sort the vector of states
       */
      
      std::sort(heights[h].begin(), heights[h].end(), LessState(*this)); 
 

      /* compute equivalent states in heights[h]
       */
      
      std::vector<int>::iterator it1 = heights[h].begin(), it2;

      while (it1 != heights[h].end()) {
        
        /* *it1 is a representant, set its nb_in to zero and adjust nb_in of the states pointed by its transitions
         */
        
	// assert(*it1 == 0 || states[*it1].nb_in > 0);

        states[*it1].nb_in = 0;

        for (typename state::trans_iterator tr = states[*it1].trans.begin(); tr != states[*it1].trans.end(); ++tr) {
          states[(*tr).second].nb_in++;
        }
 
        it2 = it1 + 1;
 
        while ((it2 != heights[h].end()) && (states[*it1] == states[*it2])) { // *it1 equiv *it2

          /*
          std::cerr << "[" << *it1 << "] == [" << *it2 << "]\n";
          std::cerr << *it1 << ": " << states[*it1] << "\n";
          std::cerr << *it2 << ": " << states[*it2] << "\n";
          */

          repr[*it2] = *it1;

          states[*it2].nb_in = 0; // should alwais stay at zero now

          ++it2;
        }
       
        it1 = it2;
      }
    }

    assert(repr[0] == 0);

    /* compute of correspondance in new numerotation
     */

    std::vector<int> corresp(states.size(), -1);
    int id = 0;

    for (int q = 0; q < corresp.size(); q++) {
      if (corresp[repr[q]] == -1) { corresp[repr[q]] = id++; }
    }



    /* fill new states table
     */

    std::vector<state> new_states(id);
   
    for (int q = 0; q < corresp.size(); q++) {
 
      int nq = corresp[q];
      
      // std::cerr << "coresp[" << q << "] = " << nq <<  " repr =" << repr[q] << "\n";

      if (nq != -1) { // q is a representant, we translate it in new states table at position nq

        if (nq > 0 && states[q].nb_in < 1) {
          std::cerr << "erreur: state " << q << " coresp = " << corresp[q]
            << "nq = " << nq << " nb_in = " << states[q].nb_in << "\n";
          std::cerr << "rerpresentant with no in transition ?\n";
          abort();
        }
        
        new_states[nq].swap(states[q]);
        
        /* remap trans to new idx */
        
        typename state::trans_iterator it;
      
        for (it = new_states[nq].trans.begin(); it != new_states[nq].trans.end(); ++it) { 
          (*it).second = corresp[repr[(*it).second]];
        }
      
      }
    }

    states.swap(new_states);

    //std::cerr << "end of minimize(), nbstates = " << states.size() << "\n";
  }


  void dump(std::ostream & os = std::cerr) const {
 
    os << "dump DIC:\n";
    os << states.size() << " states\n";

    for (int q = 0; q < states.size(); q++) {
      os << q << ": " << states[q] << "\n"; 
    }
  }

  friend std::ostream & operator<< (std::ostream & os, const dic_fsa2 & dic) {
    dic.dump(os);
    return os;
  }

  
  void dump_dot(std::ostream & os = std::cerr) const {
  
    os << 
    "# fst_dic output\n\n"
    "digraph G {\n\n"
    " graph [ center = true, orientation = landscape, rankdir = LR ];\n"
    " node  [ shape  = circle ];\n\n";

    for (int q = 0; q < states.size(); q++) {
      
      os << "  " << q << " [ label=\"" << q << ':' << states[q].card << "\" ";
      if (states[q].final) { os << "shape=doublecircle "; }
      os << "];\n";

      typename state::const_trans_iterator tr;
      for (tr = states[q].trans.begin(); tr != states[q].trans.end(); ++tr) {
        os << "  " << q << " -> " << (*tr).second << " [ label = \"" << static_cast<char>((*tr).first) << "\" ];\n";
      }
    }
  
    os << "}\n";  
  }


  friend std::ostream & operator<< (std::ostream & os, const std::set<value_t> & values) {
    os << "[ (" << values.size() << ") ";
    copy(values.begin(), values.end(), std::ostream_iterator<value_t>(os, ","));
    os << " ]";
    return os;
  }
  
  inline void dump_dot(const char * fname) const {
    std::ofstream os(fname);
    if (! os) {
      std::cerr << "erreur: dic_fsa2::dump_dot: unable to open '" << fname << "'\n";
      return;
    }
    dump_dot(os);
  }

  inline void dump_dot(const std::string & fname) const {
    dump_dot(fname.c_str());
  }
  
  template<class ostreamT>
  void dump_lexic(ostreamT & os = std::cerr, bool dump_idx = false) const {
    std::vector<char_t> word;
    dump_lexic(os, word, dump_idx, 0);
  }

  template<class funcT>
  void apply_lexic(funcT func) const {
    std::vector<char_t> word;
    apply_lexic(word, 0, 0, func);
  }


  int new_state() {
    states.push_back(state());
    return states.size() - 1;
  }

  void compute_card() {
    
    //std::cerr << "compute_card, size=" << size() << "\n";
    std::stack<int> S;
    std::vector<bool> done(size(), false);

    S.push(0);

    while (! S.empty()) {
      
      int no = S.top();

      //std::cerr << "no=" << no << '\n';

      if (done[no]) { S.pop(); continue; }

      done[no] = true;
      state & q = states[no];

      q.card = q.final ? 1 : 0;

      for (trans_iterator it = q.trans.begin(); it != q.trans.end(); ++it) {
        if (done[(*it).second]) {
          q.card += states[(*it).second].card;
        } else {
          S.push((*it).second);
          done[no] = false;
        }
      }
    }
  }

private:

  int dup_state(const state & s) {

    int res = states.size();
    states.push_back(s);

    typename state::trans_iterator it;
    for (it = states[res].trans.begin(); it != states[res].trans.end(); ++it) {
      states[(*it).second].nb_in++;
    }

    return res;
  }




  template<class ostreamT>
  void dump_lexic(ostreamT & os, std::vector<char_t> & word, bool dump_idx, int i) const {

    const state & q = states[i];

    if (q.final != -1) {
      std::copy(word.begin(), word.end(), std::ostream_iterator<char_t, char_t>(os));
      if (dump_idx) { os << ',' << q.final; }
      os << '\n';
    }

    typename state::const_trans_iterator tr;

    for (tr = q.begin(); tr != q.end(); ++tr) {
      word.push_back((*tr).first);
      dump_lexic(os, word, dump_idx, (*tr).second);
      word.pop_back();
    }
  }


  template<class funcT>
  void apply_lexic(std::vector<char_t> & word, int qno, int idx, funcT func) const {

    const state & q = states[qno];

    if (q.final) { func(word, values[idx]); idx++; }

    for (const_trans_iterator tr = q.trans.begin(); tr != q.trans.end(); ++tr) {
      word.push_back((*tr).first);
      apply_lexic(word, (*tr).second, idx, func);
      idx += states[(*tr).second].card;
      word.pop_back();
    }
  }
  

};

#endif
