#ifndef _XMLWRITER_H_
#define _XMLWRITER_H_

#include <boost/filesystem/path.hpp>
#include <string>
#include <sstream>

#include <outilex/xml.h>
#include <libxml/xmlwriter.h>


class xmlwriter {

public:

  xmlwriter() : writer(NULL) {}

  xmlwriter(const boost::filesystem::path & path, int compression = 0) : writer(NULL) {
    if (! open(path, compression)) {
      throw std::runtime_error("xmlwriter: unable to open : " + path.string());
    }
  }

  xmlwriter(std::ostream & os) { open(os); }

  ~xmlwriter() { close(); }

  bool open(const boost::filesystem::path & path, int compression = 0) {
    close();
    writer = xmlNewTextWriterFilename((const char *) path.native_file_string().c_str(),
                                      compression);
    if (writer == NULL) {
      throw std::runtime_error("xmlwriter: cannot open " + path.string());
    }
    return true;
  }

  bool open(std::ostream & os) {
    close();
    writer = xmlNewTextWriter(create_ostream_xmlOutputBuffer(os));
    if (writer == NULL) {
      throw std::runtime_error("xmlwriter: cannot open writer from ostream");
    }
    return true;
  }

  void close() {
    if (writer) {
      end_document();
      xmlFreeTextWriter(writer);
      writer = NULL;
    }
  }

  operator void *() const { return writer; }
  bool operator!() const { return writer == NULL; }


  void start_document() { xmlTextWriterStartDocument(writer, NULL, NULL, NULL); }
  void end_document() { xmlTextWriterEndDocument(writer); }


  //void start_comment() { xmlTextWriterStartComment(writer); }
  //void end_comment() { xmlTextWriterEndComment(writer); }

  void write_comment(const std::string & content) {
    xmlTextWriterWriteComment(writer, (const xmlChar *) content.c_str());
  }


  /* write element */

  void start_element(const xmlChar * name) { xmlTextWriterStartElement(writer, name); }
  void start_element(const std::string & name) {
    xmlTextWriterStartElement(writer, (const xmlChar *) name.c_str()); 
  }

  void end_element() { xmlTextWriterEndElement(writer); }


  void write_element(const xmlChar * name, const xmlChar * content) {
    xmlTextWriterWriteElement(writer, name, content);
  }
  void write_element(const std::string & name, const std::string & content) {
    xmlTextWriterWriteElement(writer, (const xmlChar *) name.c_str(),
                              (const xmlChar *) content.c_str());
  }


  /* write an empty element */

  void write_element(const xmlChar * name) { start_element(name); end_element(); }
  void write_element(const std::string & name) { start_element(name); end_element(); }


  /* raw string output */

  void write_raw(const xmlChar * raw, int len) {
    xmlTextWriterWriteRawLen(writer, raw, len);
  }
  void write_raw(const std::string & raw) {
    xmlTextWriterWriteRawLen(writer, (const xmlChar *) raw.c_str(), raw.size());
  }


  /* string (escape special char) */

  void write_string(const xmlChar * text) {
    xmlTextWriterWriteString(writer, text);
  }
  void write_string(const std::string & text) {
    xmlTextWriterWriteString(writer, (const xmlChar *) text.c_str());
  }


  /* attribute output */

  void start_attribute(const xmlChar * name) {
    xmlTextWriterStartAttribute(writer, name);
  }
  void start_attribute(const std::string & name) {
    xmlTextWriterStartAttribute(writer, (const xmlChar *) name.c_str());
  }

  void end_attribute() { xmlTextWriterEndAttribute(writer); }

  void write_attribute(const xmlChar * name, const xmlChar * content) {
    xmlTextWriterWriteAttribute(writer, name, content);
  }
  void write_attribute(const std::string & name, const std::string & content) {
    xmlTextWriterWriteAttribute(writer, (const xmlChar *) name.c_str(),
                                (const xmlChar *) content.c_str());
  }


  /* CDATA output */

  void start_CDATA() { xmlTextWriterStartCDATA(writer); }

  void end_CDATA()   { xmlTextWriterEndCDATA(writer); }

  void write_CDATA(const std::string & data) {
    xmlTextWriterWriteCDATA(writer, (const xmlChar *) data.c_str());
  }


  /* miscelanious */

  void set_indent(int indent) { xmlTextWriterSetIndent(writer, indent); }
  void flush() { xmlTextWriterFlush(writer); }

  template<typename T>
  xmlwriter & operator<<(const T & t) {
    std::ostringstream os;
    os << t; write_string(os.str());
    return *this;
  }

protected:
  xmlTextWriterPtr writer;
};

struct XML_Writer {

  xmlwriter & writer;

  XML_Writer(xmlwriter & w) : writer(w) {}

  template<typename Type>
  void operator()(const Type & a) { a.write_xml(writer); }
};

/* d for data
 */
template<typename Data>
struct XML_Writerd {

  typedef Data data_type;

  xmlwriter & writer;
  data_type data;

  XML_Writerd(xmlwriter & w, const data_type & dat) : writer(w), data(dat) {}

  template<typename Type>
  void operator()(const Type & a) { a.write_XML(writer, data); }
};

template<typename Data>
XML_Writerd<Data> make_XML_writerd(xmlwriter & writer, const Data & data) {
  return XML_Writerd<Data>(writer, data);
}

#endif
